<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\Invoice;
use App\Models\Worker;
use App\Models\Payment;
use App\Models\ExternalAgency;
use App\Models\EmploymentStatus;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Barryvdh\DomPDF\Facade\Pdf; // Include the PDF facade
use App\Models\Voucher;
use App\Models\SubAgent;
use App\Models\MonthlyExpense;
use Illuminate\Support\Facades\Mail;
use App\Mail\InvoiceMail;
use Spatie\Activitylog\Facades\LogActivity;
use Spatie\Activitylog\LogOptions;
use Spatie\Activitylog\Traits\LogsActivity;


class InvoiceController extends Controller
{
    use LogsActivity;

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly(['status', 'payment_status', 'paid_amount', 'payment_date', 'payment_method', 'payment_notes'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    public function index(Request $request)
    {
        $invoices = Invoice::query()
            ->with('externalAgency')
            ->when($request->filled('invoice_number'), fn($q) => $q->where('invoice_number', 'like', '%'.$request->invoice_number.'%'))
            ->when($request->filled('external_agency_id'), fn($q) => $q->where('external_agency_id', $request->external_agency_id))
            ->when($request->filled('status'), fn($q) => $q->where('status', $request->status))
            ->when($request->filled('payment_status'), fn($q) => $q->where('payment_status', $request->payment_status))
            ->when($request->filled('from_date'), fn($q) => $q->whereDate('invoice_date', '>=', $request->from_date))
            ->when($request->filled('to_date'), fn($q) => $q->whereDate('invoice_date', '<=', $request->to_date))
            ->when($request->filled('min_amount'), fn($q) => $q->where('total_amount', '>=', $request->min_amount))
            ->when($request->filled('max_amount'), fn($q) => $q->where('total_amount', '<=', $request->max_amount))
            ->when($request->filled('sort_by'), function ($query) use ($request) {
                $direction = $request->input('direction', 'desc');
                $columns = ['invoice_date', 'total_amount', 'created_at'];
                if (in_array($request->sort_by, $columns)) {
                    $query->orderBy($request->sort_by, $direction);
                }
            }, fn($q) => $q->orderBy('invoice_date', 'desc'))
            ->paginate($request->per_page ?? 5)
            ->withQueryString();

        $agencies = ExternalAgency::select('id', 'company_name', 'recruitment_fee', 'email', 'phone', 'city', 'country')
            ->orderBy('company_name')
            ->get()
            ->keyBy('id');

        $invoicedWorkers = Invoice::pluck('workers_data')
            ->map(function($data) {
                if (is_array($data)) {
                    return $data;
                }
                return json_decode($data, true) ?? [];
            })
            ->flatten(1)
            ->groupBy('worker_id');

        $eligibleWorkers = Worker::whereHas('employmentStatus', function($q) {
                $q->whereIn('status', [
                    EmploymentStatus::STATUS_VISA_ISSUED,
                    EmploymentStatus::STATUS_TRAVELLED
                ]);
            })
            ->with(['employmentStatus.externalAgency', 'subAgent'])
            ->get()
            ->filter(function($worker) use ($invoicedWorkers) {
                $workerInvoices = $invoicedWorkers->get($worker->id);
                if (!$workerInvoices) return true;
                
                // Check if worker has been invoiced for both fees
                $hasStampingFee = false;
                $hasArrivalFee = false;
                
                foreach ($workerInvoices as $invoice) {
                    if (!empty($invoice['stamping_fee'])) $hasStampingFee = true;
                    if (!empty($invoice['arrival_fee'])) $hasArrivalFee = true;
                }
                
                // Return true if either fee hasn't been invoiced
                return !($hasStampingFee && $hasArrivalFee);
            })
            ->map(function($worker) use ($agencies, $invoicedWorkers) {
                $agency = $worker->employmentStatus->externalAgency ?? null;
                if ($agency) {
                    $worker->contracted_agency = $agency;
                    
                    // Get the latest invoice for this worker
                    $workerInvoices = $invoicedWorkers->get($worker->id);
                    $latestInvoice = $workerInvoices ? collect($workerInvoices)->last() : null;
                    
                    // Set fee values or status
                    if ($latestInvoice) {
                        $worker->stamping_fee = !empty($latestInvoice['stamping_fee']) 
                            ? ['status' => $latestInvoice['status'] ?? 'pending', 'amount' => $latestInvoice['stamping_fee']]
                            : $agency->recruitment_fee / 2;
                            
                        $worker->arrival_fee = !empty($latestInvoice['arrival_fee'])
                            ? ['status' => $latestInvoice['status'] ?? 'pending', 'amount' => $latestInvoice['arrival_fee']]
                            : $agency->recruitment_fee / 2;
                    } else {
                        $worker->stamping_fee = $agency->recruitment_fee / 2;
                        $worker->arrival_fee = $agency->recruitment_fee / 2;
                    }
                }
                return $worker;
            })
            ->filter(fn($w) => $w->contracted_agency !== null);

        $subAgents = SubAgent::all();
        $expenses = MonthlyExpense::orderBy('created_at', 'desc')->get();
        $voucherId = $request->input('voucher_id');

        // Get vouchers with their relationships
        $vouchers = Voucher::with(['subAgent', 'workers'])
            ->latest()
            ->paginate($request->per_page ?? 5);

        // Get the active tab from the request, default to 'invoices'
        $activeTab = $request->get('tab', 'invoices');

        return view('invoices.index', [
            'invoices' => $invoices,
            'agencies' => $agencies,
            'eligibleWorkers' => $eligibleWorkers,
            'subAgents' => $subAgents,
            'voucherId' => $voucherId,
            'expenses' => $expenses,
            'vouchers' => $vouchers,
            'activeTab' => $activeTab,
            'statuses' => ['draft', 'sent', 'paid', 'overdue'],
            'payment_statuses' => ['pending', 'partial', 'paid']
        ]);
    }

    public function create()
    {
        $eligibleWorkers = Worker::whereHas('employmentStatus', function($q) {
                $q->whereIn('status', [
                    EmploymentStatus::STATUS_VISA_ISSUED,
                    EmploymentStatus::STATUS_TRAVELLED
                ]);
            })
            ->with('employmentStatus.externalAgency')
            ->get()
            ->map(function($worker) {
                $agency = $worker->employmentStatus->externalAgency ?? null;
                if (!$agency) return null;
                $worker->stamping_fee = $agency->recruitment_fee / 2;
                $worker->arrival_fee = $agency->recruitment_fee / 2;
                $worker->contracted_agency = $agency;
                return $worker;
            })
            ->filter();

        $invoiceNumber = 'INV-' . Carbon::now()->format('Ymd') . str_pad(
            Invoice::whereDate('created_at', Carbon::today())->count() + 1, 4, '0', STR_PAD_LEFT
        );

        return view('invoices.create', compact('eligibleWorkers', 'invoiceNumber'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'external_agency_id' => 'required|exists:external_agencies,id',
            'workers' => 'required|array',
            'workers.*' => 'exists:workers,id',
            'stamping_fees' => 'required|array',
            'stamping_fees.*' => 'numeric|min:0',
            'arrival_fees' => 'required|array',
            'arrival_fees.*' => 'numeric|min:0',
            'client_name' => 'required|string|max:255',
            'client_email' => 'required|email|max:255',
            'client_phone' => 'required|string|max:15',
            'client_city' => 'required|string|max:255',
            'client_country' => 'required|string|max:255',
            'total_amount' => 'required|numeric|min:0'
        ]);

        try {
            DB::beginTransaction();

            $agency = ExternalAgency::findOrFail($validated['external_agency_id']);
            $stampingTotal = array_sum($validated['stamping_fees']);
            $arrivalTotal = array_sum($validated['arrival_fees']);
            $totalAmount = $stampingTotal + $arrivalTotal;

            // Build workers_data array
            $workersData = [];
            foreach ($validated['workers'] as $index => $workerId) {
                $worker = Worker::findOrFail($workerId);
                $workersData[] = [
                    'worker_id' => $worker->id,
                    'worker_name' => $worker->full_name,
                    'passport_number' => $worker->passport_number,
                    'stamping_fee' => $validated['stamping_fees'][$index],
                    'arrival_fee' => $validated['arrival_fees'][$index],
                ];
            }

            // Create invoice
            $invoice = Invoice::create([
                'invoice_number' => 'INV-' . strtoupper(substr(uniqid(), -5)),
                'invoice_date' => now(),
                'external_agency_id' => $validated['external_agency_id'],
                'client_name' => $agency->company_name,
                'client_email' => $agency->email,
                'client_phone' => $agency->phone,
                'client_city' => $agency->city,
                'client_country' => $agency->country,
                'workers_data' => json_encode($workersData),
                'stamping_fee_total' => $stampingTotal,
                'arrival_fee_total' => $arrivalTotal,
                'total_amount' => $totalAmount,
                'status' => 'pending',
                'payment_status' => 'pending',
            ]);

            // Attach workers to the invoice
            foreach ($validated['workers'] as $index => $workerId) {
                $invoice->workers()->attach($workerId, [
                    'stamping_fee' => $validated['stamping_fees'][$index],
                    'arrival_fee' => $validated['arrival_fees'][$index],
                    'stamping_paid' => false,
                    'arrival_paid' => false,
                ]);
            }

            DB::commit();

            return redirect()->route('invoices.index')->with('success', 'Invoice created successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Invoice creation failed: ' . $e->getMessage());
            return back()->with('error', 'Failed to create invoice: ' . $e->getMessage());
        }
    }

    public function show(Invoice $invoice)
    {
        $invoice->load('externalAgency');
        $workers = is_array($invoice->workers_data) ? $invoice->workers_data : json_decode($invoice->workers_data, true);
        return view('invoices.show', compact('invoice', 'workers'));
    }

    public function edit(Invoice $invoice)
    {
        $invoice->load('externalAgency');
        $workers = is_array($invoice->workers_data) ? $invoice->workers_data : json_decode($invoice->workers_data, true) ?? [];

        $workers = array_map(function ($worker) {
            return array_merge([
                'worker_id' => $worker['worker_id'] ?? null,
                'worker_name' => $worker['worker_name'] ?? 'Unknown',
                'passport_number' => $worker['passport_number'] ?? 'N/A',
                'stamping_fee' => $worker['stamping_fee'] ?? 0,
                'arrival_fee' => $worker['arrival_fee'] ?? 0,
            ], $worker);
        }, $workers);

        $agencies = ExternalAgency::all();

        return view('invoices.edit', compact('invoice', 'workers', 'agencies'));
    }

    public function update(Request $request, Invoice $invoice)
    {
        $validated = $request->validate([
            'external_agency_id' => 'required|exists:external_agencies,id',
            'invoice_date' => 'required|date',
            'client_name' => 'required|string|max:255',
            'client_email' => 'nullable|email',
            'client_phone' => 'nullable|string|max:20',
            'client_city' => 'required|string|max:255',
            'client_country' => 'required|string|max:255',
            'stamping_fee_total' => 'required|numeric|min:0',
            'arrival_fee_total' => 'required|numeric|min:0',
            'total_amount' => 'required|numeric|min:0',
            'status' => 'required|in:draft,sent,paid,overdue',
            'workers_data' => 'required|json'
        ]);

        $invoice->update($validated);

        return redirect()->route('invoices.index')->with('success', 'Invoice updated successfully!');
    }

    public function destroy(Invoice $invoice)
    {
        try {
            DB::transaction(function () use ($invoice) {
                // Detach workers before deleting invoice
                $invoice->workers()->detach();
                $invoice->delete();
            });

            return response()->json([
                'success' => true,
                'message' => 'Invoice deleted successfully!'
            ]);
        } catch (\Exception $e) {
            \Log::error('Error deleting invoice: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error deleting invoice: ' . $e->getMessage()
            ], 500);
        }
    }

    public function markAsPaid(Invoice $invoice)
    {
        $invoice->update([
            'status' => Invoice::STATUS_PAID,
            'payment_status' => Invoice::PAYMENT_STATUS_PAID,
            'payment_date' => now(),
            'paid_amount' => $invoice->total_amount
        ]);

        // Update worker payment statuses
        foreach ($invoice->workers as $worker) {
            $invoice->workers()->updateExistingPivot($worker->id, [
                'stamping_paid' => true,
                'arrival_paid' => true,
            ]);
        }

        return back()->with('success', 'Invoice marked as paid!');
    }

    /**
     * Record a partial payment for an invoice
     */
    public function recordPayment(Request $request, Invoice $invoice)
    {
        try {
            $validated = $request->validate([
                'amount' => 'required|numeric|min:0.01|max:' . ($invoice->total_amount - $invoice->paid_amount),
                'payment_method' => 'required|string',
                'payment_notes' => 'nullable|string',
                'transaction_reference' => 'nullable|string',
                'payment_date' => 'required|date'
            ]);

            DB::beginTransaction();

            // Create payment record
            $payment = Payment::create([
                'invoice_id' => $invoice->id,
                'amount' => $validated['amount'],
                'payment_method' => $validated['payment_method'],
                'payment_notes' => $validated['payment_notes'] ?? null,
                'payment_date' => $validated['payment_date'],
                'transaction_reference' => $validated['transaction_reference'] ?? null
            ]);

            // Update invoice paid amount
            $invoice->paid_amount += $validated['amount'];
            $invoice->payment_date = $validated['payment_date'];
            $invoice->payment_method = $validated['payment_method'];
            $invoice->payment_notes = $validated['payment_notes'] ?? null;

            // Calculate payment progress
            $paymentProgress = ($invoice->paid_amount / $invoice->total_amount) * 100;

            // Update invoice status based on payment progress
            if ($paymentProgress >= 100) {
                $invoice->status = Invoice::STATUS_PAID;
                $invoice->payment_status = Invoice::PAYMENT_STATUS_PAID;
                
                // Update worker payment statuses
                foreach ($invoice->workers as $worker) {
                    $invoice->workers()->updateExistingPivot($worker->id, [
                        'stamping_paid' => true,
                        'arrival_paid' => true,
                    ]);
                }
            } elseif ($paymentProgress > 0) {
                $invoice->status = Invoice::STATUS_PARTIALLY_PAID;
                $invoice->payment_status = Invoice::PAYMENT_STATUS_PARTIAL;
            } else {
                $invoice->status = $invoice->isOverdue() ? Invoice::STATUS_OVERDUE : Invoice::STATUS_PENDING;
                $invoice->payment_status = Invoice::PAYMENT_STATUS_PENDING;
            }

            $invoice->save();

            // Log the payment
            activity()
                ->performedOn($invoice)
                ->withProperties([
                    'payment_id' => $payment->id,
                    'amount' => $validated['amount'],
                    'payment_method' => $validated['payment_method'],
                    'payment_date' => $validated['payment_date'],
                    'new_status' => $invoice->status,
                    'payment_progress' => $paymentProgress
                ])
                ->log('Payment recorded and status updated');

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Payment recorded successfully and invoice status updated.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error recording payment: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error recording payment: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Export invoice as PDF with enhanced formatting
     */
    public function export(Invoice $invoice)
    {
        $pdf = PDF::loadView('invoices.pdf', [
            'invoice' => $invoice,
            'company' => [
                'name' => config('app.name'),
                'address' => config('app.address'),
                'phone' => config('app.phone'),
                'email' => config('app.email'),
                'logo' => public_path('images/logo.png')
            ]
        ]);

        $pdf->setPaper('a4');
        $pdf->setOptions([
            'isHtml5ParserEnabled' => true,
            'isRemoteEnabled' => true
        ]);

        return $pdf->download("invoice_{$invoice->invoice_number}.pdf");
    }

    /**
     * Send invoice to client via email
     */
    public function sendToClient(Invoice $invoice)
    {
        try {
            Mail::to($invoice->client_email)
                ->send(new InvoiceMail($invoice));

            $invoice->update(['status' => Invoice::STATUS_SENT]);
            
            return back()->with('success', 'Invoice sent successfully!');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to send invoice: ' . $e->getMessage());
        }
    }

    public function expensesTab()
    {
        $expenses = MonthlyExpense::orderBy('created_at', 'desc')->paginate(10);
        return view('invoices.partials.monthly-expenses-tab', compact('expenses'));
    }

}
