<?php

namespace App\Http\Controllers;


use App\Models\JobListing;
use Illuminate\Http\Request;

class JobListingController extends Controller
{
    public function index(Request $request)
    {
        $query = Joblisting::query();
    
        if ($request->filled('search')) {
            $query->where('title', 'like', '%' . $request->search . '%')
                  ->orWhere('description', 'like', '%' . $request->search . '%');
        }
    
        if ($request->filled('location')) {
            $query->where('location', $request->location);
        }
    
        if ($request->filled('salary')) {
            $query->where('salary', '>=', $request->salary);
        }
    
        if ($request->filled('tag')) {
            switch ($request->tag) {
                case 'full-time':
                    $query->where('type', 'Full-time');
                    break;
                case 'contract':
                    $query->where('type', 'Contract');
                    break;
                case 'high-salary':
                    $query->where('salary', '>=', 10000);
                    break;
                case 'urgent':
                    $query->where('iurgent', true);
                    break;
            }
        }
    
        $jobs = $query->latest()->paginate(06);
    
        return view('jobs.index', [
            'jobs' => $jobs,
            'locations' => JobListing::distinct()->pluck('location')->filter()
        ]);
    }
    // app/Http/Controllers/JobListingController.php

public function apply($jobId)
{
    $job = JobListing::findOrFail($jobId);

    // Additional logic for application (like displaying a form or processing application)

    return view('jobs.apply', compact('job'));
}


    public function create()
    {
        return view('jobs.create');
    }

    public function store(Request $request)
    {
        try {
            \Log::info('Job creation attempt with data:', [
                'all' => $request->all(),
                'headers' => $request->headers->all(),
                'is_ajax' => $request->ajax()
            ]);

            $validated = $request->validate([
                'title' => 'required|string|max:255',
                'company' => 'required|string|max:255',
                'type' => 'required|string|max:255',
                'location' => 'required|string|max:255',
                'salary' => 'nullable|numeric|min:0',
                'currency' => 'required|string|max:10',
                'description' => 'required|string',
                'skills' => 'nullable|string',
                'posted' => 'nullable|date',
                'closing_date' => 'nullable|date|after_or_equal:posted',
                'urgent' => 'nullable|boolean',
                'highSalary' => 'nullable|boolean',
                'commission' => 'nullable|boolean',
                'status' => 'required|string|in:active,draft,closed',
            ], [
                'title.required' => 'The job title is required.',
                'company.required' => 'The company name is required.',
                'type.required' => 'The job type is required.',
                'location.required' => 'The job location is required.',
                'salary.numeric' => 'The salary must be a number.',
                'salary.min' => 'The salary cannot be negative.',
                'currency.required' => 'The currency is required.',
                'description.required' => 'The job description is required.',
                'closing_date.after_or_equal' => 'The closing date must be after or equal to the posted date.',
                'status.required' => 'The job status is required.',
                'status.in' => 'The job status must be active, draft, or closed.',
            ]);

            \Log::info('Validation passed. Validated data:', $validated);

            // Convert skills string to array
            if (!empty($validated['skills'])) {
                $validated['skills'] = array_map('trim', explode(',', $validated['skills']));
            }

            // Convert checkbox values to boolean
            $validated['urgent'] = $request->has('urgent');
            $validated['highSalary'] = $request->has('highSalary');
            $validated['commission'] = $request->has('commission');

            \Log::info('Processed data before creation:', $validated);

            // Create the job listing
            $job = JobListing::create($validated);

            \Log::info('Job created successfully:', ['job_id' => $job->id]);

            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Job created successfully.',
                    'job' => $job
                ]);
            }

            return redirect()->route('jobs.index')->with('success', 'Job created successfully.');
        } catch (\Illuminate\Validation\ValidationException $e) {
            \Log::error('Validation error:', [
                'errors' => $e->errors(),
                'input' => $request->all(),
                'headers' => $request->headers->all(),
                'is_ajax' => $request->ajax()
            ]);
            
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Please check the form for errors.',
                    'errors' => $e->errors()
                ], 422);
            }
            
            return back()->withErrors($e->errors())->withInput();
        } catch (\Exception $e) {
            \Log::error('Job creation error:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'input' => $request->all(),
                'headers' => $request->headers->all(),
                'is_ajax' => $request->ajax()
            ]);
            
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred while creating the job.',
                    'error' => $e->getMessage()
                ], 500);
            }
            
            return back()->with('error', 'An error occurred while creating the job.')->withInput();
        }
    }


    public function show(Job $job)
    {
        return view('jobs.show', compact('job'));
    }
}
