<?php

namespace App\Http\Controllers;

use App\Models\MonthlyExpense;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\MonthlyExpensesExport;
use Maatwebsite\Excel\Facades\Excel;

class MonthlyExpenseController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $expenses = MonthlyExpense::orderBy('created_at', 'desc')->paginate(5);
        return view('invoices.index', compact('expenses'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'task' => 'required|string|max:255',
            'amount_paid' => 'nullable|numeric',
            'credit' => 'nullable|numeric',
            'notes' => 'nullable|string|max:1000',
            'type' => 'required|in:expense,credit'
        ]);

        $data = [
            'name' => $request->name,
            'task' => $request->task,
            'notes' => $request->notes,
            'due_date' => now(),
        ];

        if ($request->type === 'expense') {
            $data['amount_paid'] = $request->amount_paid ?? 0;
            $data['credit'] = 0;
        } else {
            $data['credit'] = $request->credit ?? 0;
            $data['amount_paid'] = 0;
        }

        MonthlyExpense::create($data);

        return redirect()->back()->with('success', $request->type === 'expense' ? 'Expense added successfully.' : 'Credit updated successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(MonthlyExpense $monthlyExpense)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(MonthlyExpense $expense)
    {
        return view('invoices.edit-expense', compact('expense'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, MonthlyExpense $monthlyExpense)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'task' => 'required|string|max:255',
            'amount_paid' => 'required|numeric',
            'credit' => 'nullable|numeric',
            'due_date' => 'nullable|date',
            'notes' => 'nullable|string',
        ]);

        $monthlyExpense->update($validated);

        return redirect()->back()->with('success', 'Expense updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(MonthlyExpense $expense)
    {
        $expense->delete();

        return redirect()->back()->with('success', 'Expense deleted successfully.');
    }

    public function exportPdf()
    {
        $expenses = MonthlyExpense::orderBy('due_date')->get();

        // Totals
        $totalAmountPaid = $expenses->sum('amount_paid');
        $totalCredit = $expenses->sum('credit');
        $netBalance = $totalCredit - $totalAmountPaid;

        // Recalculate current accounts row by row
        $runningBalance = 0;
        $expensesWithBalance = $expenses->map(function ($expense) use (&$runningBalance) {
            $runningBalance += $expense->credit - $expense->amount_paid;
            $expense->calculated_current_account = $runningBalance;
            return $expense;
        });

        $pdf = Pdf::loadView('invoices.monthly-expenses-pdf', [
            'expenses' => $expensesWithBalance,
            'totalAmountPaid' => $totalAmountPaid,
            'totalCredit' => $totalCredit,
            'netBalance' => $netBalance,
        ]);

        return $pdf->download('monthly-expenses.pdf');
    }

    public function export()
    {
        return Excel::download(new MonthlyExpensesExport, 'monthly-expenses.xlsx');
    }
}
