<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Worker;
use App\Models\JobApplication;
use App\Models\Invoice;
use App\Models\Activity;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\UsersExport;
use PDF;

class SuperAdminController extends Controller
{
    public function dashboard(Request $request)
    {
        // Get statistics
        $totalUsers = User::count();
        $activeWorkers = Worker::where('status', 'active')->count();
        $pendingApplications = JobApplication::where('status', 'pending')->count();
        $totalInvoices = Invoice::count();

        // Get recent activities
        $recentActivities = Activity::with('causer')
            ->latest()
            ->take(10)
            ->get()
            ->map(function ($activity) {
                return (object)[
                    'description' => $activity->description,
                    'created_at' => $activity->created_at,
                    'icon' => $this->getActivityIcon($activity->description),
                    'color' => $this->getActivityColor($activity->description)
                ];
            });

        // Recent logins (last 10)
        $recentLogins = User::orderByDesc('last_login_at')
            ->whereNotNull('last_login_at')
            ->take(10)
            ->get(['id', 'name', 'email', 'role', 'last_login_at', 'last_login_ip']);

        // Get users for management
        $users = User::orderBy('created_at', 'desc')->paginate(10);

        // Get notifications (placeholder)
        $notifications = collect(); // Replace with actual notifications logic if needed

        return view('superadmin.dashboard', [
            'users' => $users,
            'roles' => User::getAvailableRoles(),
            'totalUsers' => $totalUsers,
            'activeWorkers' => $activeWorkers,
            'pendingApplications' => $pendingApplications,
            'totalInvoices' => $totalInvoices,
            'recentActivities' => $recentActivities,
            'recentLogins' => $recentLogins,
            'notifications' => $notifications
        ]);
    }

    public function updateUserRole(Request $request, User $user)
    {
        $request->validate([
            'role' => User::getRoleValidationRules()
        ]);

        try {
            $user->update(['role' => $request->role]);
            $user->logUserAction('role updated');
            return back()->with('success', 'User role updated successfully');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to update user role: ' . $e->getMessage());
        }
    }

    /**
     * Show the details of a user.
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function viewUser($id)
    {
        try {
            $user = User::findOrFail($id);
            return view('superadmin.view-user', compact('user'));
        } catch (\Exception $e) {
            return back()->with('error', 'User not found');
        }
    }

    public function deleteUser($id)
    {
        try {
            $user = User::findOrFail($id);
            
            // Prevent deleting the last superadmin
            if ($user->isSuperAdmin() && User::where('role', User::ROLE_SUPER_ADMIN)->count() <= 1) {
                return response()->json([
                    'message' => 'Cannot delete the last superadmin'
                ], 422);
            }

            $user->logUserAction('deleted');
            $user->delete();
            return response()->json(['message' => 'User deleted successfully']);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Failed to delete user: ' . $e->getMessage()
            ], 500);
        }
    }

    private function getActivityIcon($description)
    {
        if (str_contains($description, 'created')) {
            return 'fa-plus-circle';
        } elseif (str_contains($description, 'updated')) {
            return 'fa-edit';
        } elseif (str_contains($description, 'deleted')) {
            return 'fa-trash';
        } elseif (str_contains($description, 'logged in')) {
            return 'fa-sign-in-alt';
        } else {
            return 'fa-info-circle';
        }
    }

    private function getActivityColor($description)
    {
        if (str_contains($description, 'created')) {
            return 'success';
        } elseif (str_contains($description, 'updated')) {
            return 'info';
        } elseif (str_contains($description, 'deleted')) {
            return 'danger';
        } elseif (str_contains($description, 'logged in')) {
            return 'primary';
        } else {
            return 'secondary';
        }
    }

    public function exportUsers(Request $request)
    {
        $format = $request->input('format', 'csv');
        $includeLastLogin = $request->boolean('include_last_login');
        $includeRoles = $request->boolean('include_roles');
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        $query = User::query();

        // Apply date range filter if provided
        if ($startDate && $endDate) {
            $query->whereBetween('created_at', [$startDate, $endDate]);
        }

        $users = $query->get();

        // Prepare data for export
        $data = $users->map(function ($user) use ($includeLastLogin, $includeRoles) {
            $userData = [
                'ID' => $user->id,
                'Name' => $user->name,
                'Email' => $user->email,
                'Created At' => $user->created_at->format('Y-m-d H:i:s'),
            ];

            if ($includeLastLogin) {
                $userData['Last Login'] = $user->last_login_at ? $user->last_login_at->format('Y-m-d H:i:s') : 'Never';
                $userData['Last Login IP'] = $user->last_login_ip ?? 'N/A';
            }

            if ($includeRoles) {
                $userData['Role'] = $user->role;
            }

            return $userData;
        });

        switch ($format) {
            case 'csv':
                return Excel::download(new UsersExport($data), 'users.csv');
            case 'xlsx':
                return Excel::download(new UsersExport($data), 'users.xlsx');
            case 'pdf':
                $pdf = PDF::loadView('superadmin.users-pdf', ['users' => $data]);
                return $pdf->download('users.pdf');
            default:
                return back()->with('error', 'Invalid export format');
        }
    }

    public function importUsers(Request $request)
    {
        // Placeholder for user import logic
        return view('superadmin.import-users');
    }

    public function downloadTemplate()
    {
        $headers = [
            'Name',
            'Email',
            'Role',
            'Password'
        ];

        $data = collect([$headers]);

        return Excel::download(new UsersExport($data), 'users_import_template.xlsx');
    }
}