<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Worker;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class WorkerController extends Controller
{
    public function index(Request $request)
    {
        $keywords = $request->input('keywords');
        $search = $request->input('search');
        $gender = $request->input('gender');
        $location = $request->input('location');
        $education_level = $request->input('education_level');
        $perPage = $request->input('perPage', 10);
        $occupation = $request->input('occupation');
        $skill = $request->input('skill');
        $marital_status = $request->input('marital_status');
        $religion = $request->input('religion');
        $from_date = $request->input('from_date');
        $to_date = $request->input('to_date');
        $status = $request->input('status', 'All');
        $sort = $request->input('sort', 'newest');
        $passport_number = $request->input('passport_number');

        $workers = Worker::query()
            ->when($search, function ($query, $search) {
                return $query->where(function ($q) use ($search) {
                    $q->where('surname', 'like', "%{$search}%")
                      ->orWhere('given_names', 'like', "%{$search}%")
                      ->orWhere('mobile_number', 'like', "%{$search}%")
                      ->orWhere('email', 'like', "%{$search}%")
                      ->orWhere('passport_number', 'like', "%{$search}%");
                });
            })
            ->when($keywords, function ($query, $keywords) {
                return $query->where(function ($q) use ($keywords) {
                    $q->where('surname', 'like', "%{$keywords}%")
                      ->orWhere('given_names', 'like', "%{$keywords}%")
                      ->orWhere('email', 'like', "%{$keywords}%")
                      ->orWhere('mobile_number', 'like', "%{$keywords}%")
                      ->orWhere('occupation', 'like', "%{$keywords}%")
                      ->orWhere('skills', 'like', "%{$keywords}%")
                      ->orWhere('passport_number', 'like', "%{$keywords}%");
                });
            })
            ->when($passport_number, function ($query, $passport_number) {
                return $query->where('passport_number', 'like', "%{$passport_number}%");
            })
            ->when($gender, fn($query) => $query->where('gender', $gender))
            ->when($location, fn($query) => $query->where(function ($q) use ($location) {
                $q->where('city', 'like', "%{$location}%")
                  ->orWhere('country', 'like', "%{$location}%");
            }))
            ->when($education_level, fn($query) => $query->where('education_level', 'like', "%{$education_level}%"))
            ->when($occupation, fn($query) => $query->where('occupation', $occupation))
            ->when($skill, fn($query) => $query->where('skills', 'like', "%{$skill}%"))
            ->when($marital_status, fn($query) => $query->where('marital_status', $marital_status))
            ->when($religion, fn($query) => $query->where('religion', $religion))
            ->when($from_date, fn($query) => $query->whereDate('created_at', '>=', $from_date))
            ->when($to_date, fn($query) => $query->whereDate('created_at', '<=', $to_date))
            ->when($status !== 'All', function($query) use ($status) {
                // Normalize the status for database query
                $normalizedStatus = match(strtolower($status)) {
                    'back_out' => 'back_out',
                    'available' => 'available',
                    'processing' => 'processing',
                    'employee' => 'employee',
                    default => strtolower($status)
                };
                return $query->where('status', $normalizedStatus);
            })
            ->when($sort === 'oldest', fn($query) => $query->orderBy('created_at', 'asc'), 
                fn($query) => $query->orderBy('created_at', 'desc'))
            ->paginate($perPage)
            ->withQueryString();

        if ($request->ajax()) {
            return response()->json([
                'current_page' => $workers->currentPage(),
                'data' => $workers->getCollection()->map(function ($worker) {
                    return [
                        'passport_number' => $worker->passport_number,
                        'worker_name' => $worker->surname . ' ' . $worker->given_names,
                        'country' => $worker->country,
                        'status' => $worker->status,
                        'status_badge' => $worker->status_badge,
                    ];
                }),
                'first_page_url' => $workers->url(1),
                'from' => $workers->firstItem(),
                'last_page' => $workers->lastPage(),
                'last_page_url' => $workers->url($workers->lastPage()),
                'links' => $workers->linkCollection()->toArray(),
                'next_page_url' => $workers->nextPageUrl(),
                'path' => $workers->path(),
                'per_page' => $workers->perPage(),
                'prev_page_url' => $workers->previousPageUrl(),
                'to' => $workers->lastItem(),
                'total' => $workers->total(),
            ]);
        }

        $occupations = Worker::select('occupation')->distinct()->pluck('occupation')->filter();
        $skills = Worker::pluck('skills')->flatMap(function ($item) {
            return json_decode($item) ?? explode(',', $item);
        })->filter()->map(fn($s) => trim($s))->unique()->values();

        return view('workers.index', compact('workers', 'occupations', 'skills', 'status'));
    }

    public function generateCvPdf($id) 
    {
        $worker = Worker::findOrFail($id);
        $pdf = Pdf::loadView('workers.cv', compact('worker'));
        return $pdf->download('Candidate_CV.pdf');
    }

    public function generateCV($id)
    {
        $worker = Worker::findOrFail($id);
        $pdf = Pdf::loadView('workers.cv_pdf', compact('worker'));
        $fileName = 'Worker_CV_' . $worker->surname . '.pdf';
        return $pdf->download($fileName);
    }

    public function showCV($id)
    {
        $worker = Worker::findOrFail($id);
        return view('workers.cv', compact('worker'));
    }

    public function create()
    {
        return view('workers.create', [
            'statusOptions' => Worker::getStatusOptions(),
            'subAgents' => \App\Models\SubAgent::all(),
        ]);
    }
    

    public function store(Request $request)
    {
        try {
            DB::beginTransaction();

            // Enhanced validation with custom messages
            $validatedData = $request->validate([
                'surname' => 'required|string|max:255',
                'given_names' => 'required|string|max:255',
                'date_of_birth' => 'required|date',
                'gender' => 'required|string|in:Male,Female,Other',
                'marital_status' => 'required|string|in:Single,Married,Divorced,Widowed',
                'religion' => 'required|string',
                'occupation' => 'required|string',
                'education_level' => 'required|string',
                'work_experience' => 'required|string',
                'skills' => 'required|array',
                'languages' => 'required|array',
                'photo_url' => 'required|image|mimes:jpeg,png,jpg|max:2048',
                'passport_number' => [
                    'required', 'string', 'max:255',
                    Rule::unique('workers', 'passport_number')
                ],
                'passport_issue_place' => 'required|string|max:255',
                'passport_issue_date' => 'required|date',
                'passport_expiry_date' => 'required|date',
                'passport_file' => 'required|file|mimes:pdf,jpeg,png|max:2048',
                'country' => 'required|string|max:255',
                'city' => 'required|string|max:255',
                'place_of_birth' => 'required|string|max:255',
                'mobile_number' => 'required|string|max:20',
                'email' => [
                    'required', 'email', 'max:255',
                    Rule::unique('workers', 'email')
                ],
                'id_number' => [
                    'required', 'string', 'max:255',
                    Rule::unique('workers', 'id_number')
                ],
                'sub_agent_id' => 'nullable|exists:sub_agents,id',
                'relative_name' => 'required|string|max:255',
                'relative_kinship' => 'required|string|max:255',
                'relative_phone' => 'required|string|max:20',
                'relative_address' => 'required|string|max:255',
                'acknowledge' => 'required|boolean',
            ], [
                'required' => 'The :attribute field is required',
                'accepted' => 'You must accept the acknowledgment',
                'unique' => 'The :attribute is already in use',
                'in' => 'The selected :attribute is invalid',
                '*.mimes' => 'Invalid file type for :attribute',
                '*.max' => 'The :attribute must not exceed :max kilobytes'
            ]);

            // Generate unique filenames
            $photoExtension = $request->file('photo_url')->getClientOriginalExtension();
            $passportExtension = $request->file('passport_file')->getClientOriginalExtension();
            
            $photoFileName = 'photo_' . uniqid() . '_' . time() . '.' . $photoExtension;
            $passportFileName = 'passport_' . uniqid() . '_' . time() . '.' . $passportExtension;

            // File upload handling with custom filenames
            $photoPath = '/images/worker_photos/' . $photoFileName;
            $passportPath = '/images/worker_passports/' . $passportFileName;

            // Move files to their respective directories
            $request->file('photo_url')->move(public_path('images/worker_photos'), $photoFileName);
            $request->file('passport_file')->move(public_path('images/worker_passports'), $passportFileName);

            // Prepare data
            $workerData = array_merge($validatedData, [
                'photo_url' => $photoPath,
                'passport_file' => $passportPath,
                'skills' => json_encode($validatedData['skills']),
                'languages' => json_encode($validatedData['languages']),
                'status' => 'available'
            ]);

            // Create worker record
            $worker = Worker::create($workerData);

            DB::commit();

            if ($request->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Worker created successfully!',
                    'redirect' => route('workers.index')
                ]);
            }

            return redirect()->route('workers.index')
                ->with('success', 'Worker created successfully!');

        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            if ($request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $e->errors()
                ], 422);
            }
            return back()
                ->withErrors($e->validator)
                ->withInput();
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Worker creation failed: ' . $e->getMessage());
            if ($request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Worker creation failed: ' . $e->getMessage()
                ], 500);
            }
            return back()
                ->with('error', 'Worker creation failed. Please try again.')
                ->withInput();
        }
    }

    public function show(Worker $worker)
    {
        return view('workers.show', compact('worker'));
    }

    public function edit(Worker $worker)
    {
        return view('workers.edit', [
            'worker' => $worker,
            'statusOptions' => Worker::getStatusOptions()
        ]);
    }

    public function update(Request $request, Worker $worker)
    {
        try {
            DB::beginTransaction();

            $validated = $request->validate([
                'surname' => 'required|string|max:255',
                'given_names' => 'required|string|max:255',
                'date_of_birth' => 'required|date',
                'gender' => 'required|string|in:Male,Female,Other',
                'marital_status' => 'required|string|in:Single,Married,Divorced,Widowed',
                'religion' => 'required|string',
                'occupation' => 'required|string',
                'education_level' => 'required|string',
                'skills' => 'required|array',
                'languages' => 'required|array',
                'id_number' => 'required|string|unique:workers,id_number,' . $worker->id,
                'mobile_number' => 'required|string|max:20',
                'email' => 'required|email|unique:workers,email,' . $worker->id,
                'work_experience' => 'required|string',
                'passport_number' => 'required|string|unique:workers,passport_number,' . $worker->id,
                'passport_issue_place' => 'required|string|max:255',
                'passport_issue_date' => 'required|date',
                'passport_expiry_date' => 'required|date',
                'country' => 'required|string|max:255',
                'city' => 'required|string|max:255',
                'place_of_birth' => 'required|string|max:255',
                'relative_name' => 'required|string|max:255',
                'relative_kinship' => 'required|string|max:255',
                'relative_phone' => 'required|string|max:20',
                'relative_address' => 'required|string|max:255',
                'passport_file' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048',
                'photo_url' => 'nullable|file|mimes:jpg,jpeg,png|max:2048',
                'acknowledge' => 'required|boolean',
                'backout_reason' => 'nullable|string|max:500',
            ], [
                'required' => 'The :attribute field is required',
                'accepted' => 'You must accept the acknowledgment',
                'unique' => 'The :attribute is already in use',
                'in' => 'The selected :attribute is invalid',
                '*.mimes' => 'Invalid file type for :attribute',
                '*.max' => 'The :attribute must not exceed :max kilobytes'
            ]);

            if ($request->hasFile('passport_file')) {
                // Delete old file if exists
                if ($worker->passport_file) {
                    $oldPath = public_path($worker->passport_file);
                    if (file_exists($oldPath)) {
                        unlink($oldPath);
                    }
                }
                
                // Generate unique filename
                $passportExtension = $request->file('passport_file')->getClientOriginalExtension();
                $passportFileName = 'passport_' . uniqid() . '_' . time() . '.' . $passportExtension;
                $validated['passport_file'] = '/images/worker_passports/' . $passportFileName;
                
                // Move file to directory
                $request->file('passport_file')->move(public_path('images/worker_passports'), $passportFileName);
            }

            if ($request->hasFile('photo_url')) {
                // Delete old file if exists
                if ($worker->photo_url) {
                    $oldPath = public_path($worker->photo_url);
                    if (file_exists($oldPath)) {
                        unlink($oldPath);
                    }
                }
                
                // Generate unique filename
                $photoExtension = $request->file('photo_url')->getClientOriginalExtension();
                $photoFileName = 'photo_' . uniqid() . '_' . time() . '.' . $photoExtension;
                $validated['photo_url'] = '/images/worker_photos/' . $photoFileName;
                
                // Move file to directory
                $request->file('photo_url')->move(public_path('images/worker_photos'), $photoFileName);
            }

            // Convert arrays to JSON
            $validated['skills'] = json_encode($validated['skills']);
            $validated['languages'] = json_encode($validated['languages']);

            // Keep existing status if not changed
            if (!isset($validated['status'])) {
                $validated['status'] = $worker->status;
            }

            $worker->update($validated);

            DB::commit();

            return redirect()->route('workers.index')
                ->with('success', 'Worker updated successfully!');

        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return back()
                ->withErrors($e->validator)
                ->withInput();
            
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Worker update failed: ' . $e->getMessage());
            return back()
                ->with('error', 'Worker update failed. Please try again.')
                ->withInput();
        }
    }

    public function destroy(Worker $worker)
    {
        // Delete associated files
        if ($worker->photo_url) {
            $oldPath = public_path($worker->photo_url);
            if (file_exists($oldPath)) {
                unlink($oldPath);
            }
        }
        if ($worker->passport_file) {
            $oldPath = public_path($worker->passport_file);
            if (file_exists($oldPath)) {
                unlink($oldPath);
            }
        }

        $worker->delete();

        return redirect()->route('workers.index')
            ->with('success', 'Worker deleted successfully!');
    }

    // Additional status management methods
    public function markAsavailable(Worker $worker)
    {
        $worker->markAsAvailable();
        return back()->with('success', 'Worker marked as available');
    }

    public function markAsprocessing(Worker $worker)
    {
        $worker->markAsprocessing();
        return back()->with('success', 'Worker marked as processing');
    }

    // Change markAsEmployed to markAsEmployee
    public function markAsemployee(Worker $worker)
    {
        $worker->update(['status' => 'employee']);
        return back()->with('success', 'Worker marked as employee');
    }

    public function markAsback_out(Request $request, Worker $worker)
    {
        $validated = $request->validate([
            'backout_reason' => 'nullable|string|max:500'
        ]);

        $worker->markAsBackOut($validated['backout_reason'] ?? null);

        return back()->with('success', 'Worker marked as back out');
    }

    public function revokeBackout(Worker $worker)
    {
        try {
            if (strtolower($worker->status) !== 'back_out') {
                return redirect()->back()->with('error', 'This worker is not marked as back out.');
            }

            $worker->update([
                'status' => 'available',
                'backout_reason' => null
            ]);

            return redirect()->back()->with('success', [
                'title' => 'Status Updated',
                'message' => 'Worker status has been changed back to available.'
            ]);
        } catch (\Exception $e) {
            \Log::error('Error revoking back out status: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to update worker status. Please try again.');
        }
    }
}