<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class EmploymentStatus extends Model
{
    use HasFactory;

    // Employment status constants
    const STATUS_ALL = 'All';
    const STATUS_PENDING = 'Pending';
    const STATUS_SENT_NEA = 'Sent NEA';
    const STATUS_NEA_APPROVED = 'NEA Approved';
    const STATUS_MEDICAL_FIT = 'Medical Fit';
    const STATUS_READY_FOR_SUBMISSION = 'Ready for Submission';
    const STATUS_SUBMITTED_EMBASSY = 'Submitted Embassy';
    const STATUS_VISA_ISSUED = 'Visa Issued';
    const STATUS_TRAVELLED = 'Travelled';
    const STATUS_CANCELED = 'Canceled';

    /**
     * Default model attributes
     */
    protected $attributes = [
        'status' => self::STATUS_PENDING,
        'cancellation_reason' => null, // ✅ Optional default

    ];

    protected $fillable = [
        'contract_number',
        'visa_number',
        'worker_id',
        'payment_status',
        'status',
        'creation_date',
        'external_agency_id',
        'cancellation_reason',
        'cancelled_by',
    ];

    /**
     * The "booted" method of the model.
     */
    protected static function booted()
    {
        static::updated(function ($employmentStatus) {
            if ($employmentStatus->isDirty('status') && $employmentStatus->status === self::STATUS_TRAVELLED) {
                $employmentStatus->worker->update(['status' => 'employee']);
            }
        });
    }

    /**
     * Relationship with the Worker model.
     */
    public function worker()
    {
        return $this->belongsTo(Worker::class, 'worker_id');
    }

    public function invoice()
    {
        return $this->hasOne(Invoice::class);
    }

    /**
     * Relationship with the ExternalAgency model.
     */
    public function externalAgency()
    {
        return $this->belongsTo(ExternalAgency::class, 'external_agency_id');
    }

    /**
     * Accessor to format the creation date to a readable format.
     */
    public function getCreationDateAttribute($value)
    {
        return Carbon::parse($value)->format('d M Y');
    }

    public function cancelledByUser()
    {
        return $this->belongsTo(User::class, 'cancelled_by');
    }
}
