<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Spatie\Activitylog\LogOptions;
use Spatie\Activitylog\Traits\LogsActivity;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Models\Traits\SystemActivityLog;

class Invoice extends Model
{
    use HasFactory, LogsActivity, SoftDeletes, SystemActivityLog;

    // Status Constants
    const STATUS_DRAFT = 'draft';
    const STATUS_PENDING = 'pending';
    const STATUS_PARTIALLY_PAID = 'partially_paid';
    const STATUS_PAID = 'paid';
    const STATUS_OVERDUE = 'overdue';
    const STATUS_CANCELLED = 'cancelled';

    // Payment Status Constants
    const PAYMENT_STATUS_PENDING = 'pending';
    const PAYMENT_STATUS_PARTIAL = 'partial';
    const PAYMENT_STATUS_PAID = 'paid';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<string>
     */
    // In app/Models/Invoice.php
protected $fillable = [
    'external_agency_id', // Update this
    'invoice_number',
    'invoice_date',
    'client_name',
    'client_email', 
    'client_phone',
    'client_city',
    'client_country',
    'stamping_fee_total',
    'arrival_fee_total',
    'total_amount',
    'status',
    'payment_status',
    'payment_date',
    'due_date',
    'workers_data', // Must be included!
    'notes'
];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'invoice_date' => 'date',
        'payment_date' => 'date',
        'due_date' => 'date',
        'workers_data' => 'array',
        'total_amount' => 'decimal:2'
    ];

    /**
     * Boot the model.
     */
    protected static function boot(): void
    {
        parent::boot();

        static::creating(function (self $invoice) {
            $invoice->generateInvoiceNumber();
            $invoice->setDefaultStatus();
        });
    }

    /**
     * Generate a sequential invoice number.
     */
    protected function generateInvoiceNumber(): void
    {
        if (empty($this->invoice_number)) {
            $latestId = (int) self::max('id') ?? 0;
            $this->invoice_number = 'INV-' . str_pad($latestId + 1, 5, '0', STR_PAD_LEFT);
        }
    }

    /**
     * Set default status if not provided.
     */
    protected function setDefaultStatus(): void
    {
        if (empty($this->status)) {
            $this->status = 'draft';
        }
    }

    /**
     * Get the employment status associated with the invoice.
     */
    public function employmentStatus(): BelongsTo
    {
        return $this->belongsTo(EmploymentStatus::class);
    }

    /**
     * Calculate the total amount from fees.
     */
    public function calculateTotal(): void
    {
        $this->total_amount = $this->stamping_fee_total + $this->arrival_fee_total;
    }

    /**
     * Get the external agency that owns the invoice.
     */
    public function externalAgency(): BelongsTo
    {
        return $this->belongsTo(ExternalAgency::class, 'external_agency_id');
    }

    public function workers()
    {
        return $this->belongsToMany(Worker::class)
            ->withPivot('stamping_fee', 'arrival_fee', 'stamping_paid', 'arrival_paid')
            ->withTimestamps();
    }

    public function payments()
    {
        return $this->hasMany(Payment::class);
    }

    /**
     * Check if invoice is overdue
     */
    public function isOverdue(): bool
    {
        return $this->status !== self::STATUS_PAID && 
               $this->due_date && 
               $this->due_date->isPast();
    }

    /**
     * Calculate payment progress
     */
    public function getPaymentProgress(): float
    {
        if ($this->total_amount === 0) return 0;
        return ($this->paid_amount / $this->total_amount) * 100;
    }

    /**
     * Update invoice status based on payment
     */
    public function updatePaymentStatus(): void
    {
        $progress = $this->getPaymentProgress();
        
        if ($progress === 100) {
            $this->status = self::STATUS_PAID;
            $this->payment_status = self::PAYMENT_STATUS_PAID;
        } elseif ($progress > 0) {
            $this->status = self::STATUS_PARTIALLY_PAID;
            $this->payment_status = self::PAYMENT_STATUS_PARTIAL;
        } else {
            $this->status = $this->isOverdue() ? self::STATUS_OVERDUE : self::STATUS_PENDING;
            $this->payment_status = self::PAYMENT_STATUS_PENDING;
        }

        $this->save();
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly(['status', 'payment_status', 'paid_amount', 'payment_date', 'payment_method', 'payment_notes'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

}