<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use App\Models\Traits\SystemActivityLog;

class Voucher extends Model
{
    use HasFactory, SystemActivityLog;

    const TYPE_ADVANCE = 'advance';
    const TYPE_FINAL = 'final';

    const STATUS_PENDING = 'pending';
    const STATUS_APPROVED = 'approved';
    const STATUS_REJECTED = 'rejected';
    const STATUS_PAID = 'paid';

    protected $fillable = [
        'sub_agent_id',
        'voucher_type',
        'total_amount',
        'status',
        'notes',
        'voucher_number'
    ];

    protected $casts = [
        'total_amount' => 'decimal:2',
    ];

    /**
     * Get the sub-agent associated with the voucher.
     */
    public function subAgent(): BelongsTo
    {
        return $this->belongsTo(SubAgent::class);
    }

    /**
     * Get the workers associated with the voucher.
     */
    public function workers(): BelongsToMany
    {
        return $this->belongsToMany(Worker::class)
            ->withPivot('amount')
            ->withTimestamps();
    }

    /**
     * Get the formatted voucher number.
     */
    public function getVoucherNumberAttribute(): string
    {
        return 'VCH-' . str_pad($this->id, 6, '0', STR_PAD_LEFT);
    }

    /**
     * Check if the voucher can be edited.
     */
    public function canBeEdited(): bool
    {
        return in_array($this->status, [self::STATUS_PENDING]);
    }

    /**
     * Check if the voucher can be cancelled.
     */
    public function canBeCancelled(): bool
    {
        return in_array($this->status, [self::STATUS_PENDING, self::STATUS_APPROVED]);
    }

    /**
     * Check if the voucher can be approved.
     */
    public function canBeApproved(): bool
    {
        return $this->status === self::STATUS_PENDING;
    }

    /**
     * Check if the voucher can be rejected.
     */
    public function canBeRejected(): bool
    {
        return $this->status === self::STATUS_PENDING;
    }

    /**
     * Check if the voucher can be marked as paid.
     */
    public function canBeMarkedAsPaid(): bool
    {
        return $this->status === self::STATUS_APPROVED;
    }

    /**
     * Get the status badge class.
     */
    public function getStatusBadgeClass(): string
    {
        return match($this->status) {
            self::STATUS_PENDING => 'warning',
            self::STATUS_APPROVED => 'info',
            self::STATUS_REJECTED => 'danger',
            self::STATUS_PAID => 'success',
            default => 'secondary'
        };
    }

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($voucher) {
            $voucher->voucher_number = 'VCH-' . str_pad(static::max('id') + 1, 6, '0', STR_PAD_LEFT);
        });
    }

    
}
