<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Models\Traits\SystemActivityLog;

class Worker extends Model
{
    use HasFactory, SystemActivityLog;

    // Status constants
    const STATUS_AVAILABLE = 'available';
    const STATUS_PROCESSING = 'processing';
    const STATUS_EMPLOYEE = 'employee';
    const STATUS_BACK_OUT = 'back_out';

    protected $fillable = [
        'surname',
        'given_names',
        'date_of_birth',
        'gender',
        'marital_status',
        'religion',
        'occupation',
        'id_number',
        'education_level',
        'work_experience',
        'skills',
        'languages',
        'photo_url',
        'passport_file',
        'passport_number',
        'passport_issue_place',
        'passport_issue_date',
        'passport_expiry_date',
        'country',
        'city',
        'place_of_birth',
        'mobile_number',
        'email',
        'relative_name',
        'relative_kinship',
        'relative_phone',
        'relative_address',
        'acknowledge',
        'status',
        'backout_reason',
        'sub_agent_id',
    ];

    protected $casts = [
        'skills' => 'array',
        'languages' => 'array',
        'acknowledge' => 'boolean',
        'date_of_birth' => 'date',
        'passport_issue_date' => 'date',
        'passport_expiry_date' => 'date',
    ];

    protected $appends = ['status_badge', 'full_name'];

    /**
     * Get all available status options
     */
    public static function getStatusOptions(): array
    {
        return [
            self::STATUS_AVAILABLE => 'Available',
            self::STATUS_PROCESSING => 'Processing',
            self::STATUS_EMPLOYEE => 'Employee',
            self::STATUS_BACK_OUT => 'Back Out',
        ];
    }

    /**
     * Relationship with the EmploymentStatus model
     */
    public function employmentStatus()
    {
        return $this->hasOne(EmploymentStatus::class);
    }

    /**
     * Get HTML badge for status display
     */
    public function getStatusBadgeAttribute(): string
    {
        $status = $this->status ?? self::STATUS_AVAILABLE;
        
        return sprintf(
            '<span class="badge bg-%s">%s</span>',
            match($status) {
                self::STATUS_AVAILABLE => 'success',
                self::STATUS_PROCESSING => 'warning',
                self::STATUS_EMPLOYEE => 'primary',
                self::STATUS_BACK_OUT => 'danger',
                default => 'secondary'
            },
            self::getStatusOptions()[$status] ?? 'Unknown'
        );
    }

    /**
     * Get worker's full name
     */
    public function getFullNameAttribute(): string
    {
        return $this->given_names . ' ' . $this->surname;
    }

    /**
     * Status management methods
     */
    public function markAsAvailable(): void
    {
        $this->update([
            'status' => self::STATUS_AVAILABLE,
            'backout_reason' => null
        ]);
    }

    public function markAsProcessing(): void
    {
        $this->update([
            'status' => self::STATUS_PROCESSING,
            'backout_reason' => null
        ]);
    }

    public function markAsEmployee(): void
    {
        $this->update([
            'status' => self::STATUS_EMPLOYEE,
            'backout_reason' => null
        ]);
    }

    public function markAsBackOut(?string $reason = null): void
    {
        $this->update([
            'status' => self::STATUS_BACK_OUT,
            'backout_reason' => $reason
        ]);
    }

    public function externalAgency()
    {
        return $this->belongsTo(ExternalAgency::class);
    }

    public function invoices()
    {
        return $this->belongsToMany(Invoice::class)
            ->withPivot('stamping_fee', 'arrival_fee', 'stamping_paid', 'arrival_paid')
            ->withTimestamps();
    }

    public function subAgent()
    {
        return $this->belongsTo(SubAgent::class);
    }

    public function vouchers()
    {
        return $this->hasMany(Voucher::class);
    }

    public function worker(): BelongsTo
    {
        return $this->belongsTo(Worker::class);
    }
}