@php
    use App\Models\EmploymentStatus;
@endphp

<x-app-layout>
    <x-slot name="header">
        <div class="flex items-center justify-between">
            <h2 class="font-semibold text-xl text-gray-800 leading-tight">
                Employment Status
            </h2>

            <a href="{{ route('employment_status.create') }}" class="btn btn-success">
                <i class="fas fa-plus"></i> Assign Contract
            </a>
        </div>
    </x-slot>

    <div class="py-6">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">

            {{-- Search + Filters --}}
            <div class="card shadow-sm border-light rounded mb-4">
                <div class="card-header bg-info text-white text-center">
                    <h5>Filter Employment Status</h5>
                </div>

                <form method="GET" action="{{ route('employment_status.index') }}" class="p-3">
                    <div class="row g-2 align-items-end">
                    {{-- Passport Number --}}
                    <div class="col-md">
                        <label class="form-label">Passport Number</label>
                        <div class="input-group">
                            <span class="input-group-text"><i class="bi bi-passport"></i></span> {{-- Optional: different icon --}}
                            <input type="text" name="passport_number" class="form-control" placeholder="Passport Number" value="{{ request('passport_number') }}">
                        </div>
                    </div>

                        {{-- Visa Number --}}
                        <div class="col-md">
                            <label class="form-label">Visa Number</label>
                            <div class="input-group">
                                <span class="input-group-text"><i class="bi bi-file-earmark-person"></i></span>
                                <input type="text" name="visa_number" class="form-control" placeholder="Visa Number" value="{{ request('visa_number') }}">
                            </div>
                        </div>

                        {{-- Status --}}
                        <div class="col-md">
                            <label class="form-label">Status</label>
                            <div class="input-group">
                                <span class="input-group-text"><i class="bi bi-check-circle"></i></span>
                                <select name="status" class="form-select">
                                    <option value="">Filter by Status</option>
                                    @php
                                        $statuses = [
                                            'Pending',
                                            'Sent NEA',                                            
                                            'NEA Approved',
                                            'Medical Fit',
                                            'Ready for Submission',
                                            'Submited Embassy',
                                            'Visa Issued',
                                            'Travelled ',
                                        ];
                                    @endphp
                                    @foreach ($statuses as $status)
                                        <option value="{{ $status }}" {{ request('status') == $status ? 'selected' : '' }}>
                                            {{ $status }}
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                        </div>


                        {{-- Payment Status --}}
                        <div class="col-md">
                            <label class="form-label">Payment Status</label>
                            <div class="input-group">
                                <span class="input-group-text"><i class="bi bi-credit-card"></i></span>
                                <select name="payment_status" class="form-select">
                                    <option value="">Payment Status</option>
                                    <option value="Paid" {{ request('payment_status') == 'Paid' ? 'selected' : '' }}>Paid</option>
                                    <option value="Pending" {{ request('payment_status') == 'Pending' ? 'selected' : '' }}>Pending</option>
                                </select>
                            </div>
                        </div>

                        {{-- Submit & Clear --}}
                        <div class="col-auto">
                            <button type="submit" class="btn btn-info mt-2">
                                <i class="bi bi-funnel-fill"></i> Filter
                            </button>
                            <a href="{{ route('employment_status.index') }}" class="btn btn-outline-secondary mt-2 ms-2">
                                <i class="bi bi-x-circle"></i> Clear
                            </a>
                        </div>
                    </div>
                </form>
            </div>

            {{-- Status Tabs --}}
                @php
                    $currentStatus = request('status', 'All'); // Default status is 'All' if no status is selected
                    $queryParams = request()->except('status', 'page'); // Get all current query parameters, except 'status' and 'page'
                @endphp
                <ul class="nav nav-tabs mb-3" id="statusTabs">
                    @foreach (['All','Pending', 'Sent NEA','NEA Approved', 'Medical Fit', 'Ready for Submission', 'Submitted Embassy', 'Visa Issued', 'Travelled', 'Canceled'] as $status)
                        <li class="nav-item">
                            <a class="nav-link {{ $currentStatus === $status ? 'active' : '' }}"
                            href="{{ route('employment_status.index', $status === 'All' ? $queryParams : array_merge($queryParams, ['status' => $status])) }}">
                                {{ $status }}
                            </a>
                        </li>
                    @endforeach
                </ul>

            {{-- Employment Status List --}}
            <div class="card shadow-sm border-light rounded">
                <div class="card-header bg-light">
                    <h5 class="mb-0">Employment Status List</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        @foreach ($EmploymentStatus as $status)
                            <div class="col-12 mb-4 employment-card" data-status="{{ $status->status }}">
                                <div class="card shadow rounded shadow-lg">
                                    <div class="card-body">
                                        <div class="row align-items-start">
                                            {{-- Status + Optional Icon --}}
                                                <div class="col-auto d-flex flex-column align-items-center justify-content-center" style="min-width: 80px;">
                                                @php
                                                    $cleanStatus = trim($status->status);
                                                    $badgeColor = match ($cleanStatus) {
                                                        'Pending' => 'secondary',
                                                        'Sent NEA' => 'secondary',
                                                        'NEA Approved' => 'primary',
                                                        'Medical Fit' => 'warning text-dark',
                                                        'Ready for Submission' => 'dark',
                                                        'Submitted Embassy' => 'info',
                                                        'Visa Issued', 'Travelled' => 'success',
                                                        default => 'secondary',
                                                    };
                                                @endphp


                                                    <div class="mb-2">
                                                        <span class="badge bg-{{ $badgeColor }} text-uppercase" style="font-size: 0.6rem; line-height: 1.0; padding: 0.2rem 0.4rem;">
                                                            {{ strtoupper($cleanStatus ?? 'N/A') }}
                                                        </span>
                                                    </div>

                                                {{-- Optional Avatar --}}
                                                <img src="{{ $status->worker->photo_url ?? asset('default-avatar.png') }}"
                                                     alt="Photo of {{ $status->worker->full_name ?? 'N/A' }}"
                                                     class="rounded-circle"
                                                     style="width: 80px; height: 80px; object-fit: cover;">
                                            </div>

                                            {{-- Info --}}
                                            <div class="col">
                                                <div class="row g-3">
                                                    <div class="col-6 col-md-2">
                                                        <div class="text-muted small">Contract #</div>
                                                        <strong>{{ $status->contract_number }}</strong>
                                                    </div>
                                                    <div class="col-6 col-md-2">
                                                        <div class="text-muted small">Visa</div>
                                                        {{ $status->visa_number ?? 'N/A' }}
                                                    </div>
                                                    
                                                    <div class="col-6 col-md-2">
                                                    <div class="text-muted small">Candidate</div>
                                                        @if ($status->worker)
                                                            <a href="{{ route('workers.show', $status->worker->id) }}" class="text-primary">
                                                            <strong> {{ $status->worker->surname }} {{ $status->worker->given_names }}</strong>
                                                            </a>
                                                        @else
                                                            N/A
                                                        @endif
                                                </div>

                                                <div class="col-6 col-md-2">
                                                    <div class="text-muted small">Passport No</div>
                                                    {{ $status->worker->passport_number ?? 'N/A' }}  {{-- Assuming 'passport_number' is the field --}}
                                                </div>

                                                <div class="col-6 col-md-2">
                                                    <div class="text-muted small">Occupation</div>
                                                    {{ $status->worker->occupation ?? 'N/A' }}
                                                </div>

                                                <div class="col-6 col-md-2">
                                                    <div class="text-muted small">e_Agency</div>
                                                    {{ $status->externalAgency->company_name ?? 'N/A' }}
                                                </div>

                                                </div>

                                                <div class="row mt-3 g-3">
                                                    <div class="col-6 col-md-2">
                                                        <div class="text-muted small">Nationality</div>
                                                        {{ $status->worker->country ?? 'N/A' }}
                                                    </div>
                                                    <div class="col-6 col-md-2">
                                                        <div class="text-muted small">Payment</div>
                                                        <span class="badge bg-{{ $status->payment_status == 'Paid' ? 'success' : 'warning text-dark' }}">
                                                            {{ $status->payment_status }}
                                                        </span>
                                                    </div>
                                                    <div class="col-6 col-md-2">
                                                        <div class="text-muted small">Created on</div>
                                                        {{ \Carbon\Carbon::parse($status->created_at)->format('d M Y') }}
                                                    </div>
                                                </div>
                                            </div>

                                            {{-- Actions --}}
                                            <div class="col-auto text-end">
                                                <div class="d-flex flex-column gap-1">
                                                @php
                                                            // Normalize status values
                                                            $normalize = [
                                                                'pending' => EmploymentStatus::STATUS_PENDING,
                                                                'sent nea' => EmploymentStatus::STATUS_SENT_NEA,
                                                                'nea approved' => EmploymentStatus::STATUS_NEA_APPROVED,
                                                                'medical fit' => EmploymentStatus::STATUS_MEDICAL_FIT,
                                                                'ready for submission' => EmploymentStatus::STATUS_READY_FOR_SUBMISSION,
                                                                'submitted embassy' => EmploymentStatus::STATUS_SUBMITTED_EMBASSY,
                                                                'visa issued' => EmploymentStatus::STATUS_VISA_ISSUED,
                                                                'travelled' => EmploymentStatus::STATUS_TRAVELLED,
                                                                'canceled' => EmploymentStatus::STATUS_CANCELED,
                                                            ];

                                                            $statusRaw = trim($status->status);
                                                            $statusKey = $normalize[strtolower($statusRaw)] ?? $statusRaw;

                                                            $statusMap = [
                                                                EmploymentStatus::STATUS_PENDING => 'Send NEA',
                                                                EmploymentStatus::STATUS_SENT_NEA => 'Mark as NEA Approved',
                                                                EmploymentStatus::STATUS_NEA_APPROVED => 'Mark as Medical Fit',
                                                                EmploymentStatus::STATUS_MEDICAL_FIT => 'Mark as Ready for Submission',
                                                                EmploymentStatus::STATUS_READY_FOR_SUBMISSION => 'Mark as Submitted to Embassy',
                                                                EmploymentStatus::STATUS_SUBMITTED_EMBASSY => 'Mark as Visa Issued',
                                                                EmploymentStatus::STATUS_VISA_ISSUED => 'Mark as Travelled',
                                                                EmploymentStatus::STATUS_TRAVELLED => 'Travel Complete',
                                                            ];

                                                            $isFinal = $statusKey === EmploymentStatus::STATUS_TRAVELLED;
                                                            $isCanceled = $statusKey === EmploymentStatus::STATUS_CANCELED;
                                                            $buttonText = $statusMap[$statusKey] ?? 'Update Status';
                                                        @endphp

                                                        @if ($isFinal)
                                                            <button class="btn btn-secondary btn-sm" disabled>{{ $buttonText }}</button>
                                                        @elseif ($isCanceled)
                                                        <button class="bg-red-600 hover:bg-red-700 text-white font-semibold py-1 px-3 rounded text-sm flex items-center gap-1" disabled>
                                                            <i class="fas fa-ban"></i> Cancelled
                                                        </button>
                                                        @else
                                                            <form action="/employment_status/{{ $status->id }}/advance" method="POST" class="d-inline">
                                                                @csrf
                                                                <button type="submit" class="btn btn-link text-info text-decoration-underline p-0 m-0 btn-sm">{{ $buttonText }}</button>
                                                            </form>
                                                        @endif


                                                        @if ($status->status !== \App\Models\EmploymentStatus::STATUS_CANCELED)
                                                            <a href="{{ route('employment_status.edit', $status->id) }}" class="btn btn-info btn-sm">Edit</a>
                                                        @endif
                                                        
                                                        @php
                                                        $isCancelled = strtolower(trim($status->status)) === strtolower(\App\Models\EmploymentStatus::STATUS_CANCELED);
                                                    @endphp

                                                    @if ($isCancelled)
                                                        <!-- View Cancellation Button -->
                                                        <button class="btn btn-link text-danger text-decoration-underline p-0 m-0 btn-sm" onclick="showCancellationDetails('{{ $status->worker->given_names ?? 'N/A' }}', '{{ $status->status }}', '{{ $status->cancelledByUser->name ?? 'System/Admin' }}', '{{ $status->updated_at->format('d M Y, h:i A') }}', '{{ $status->cancellation_reason ?? 'N/A' }}')">
                                                            View Cancellation
                                                        </button>
                                                    @else
                                                       <!-- Cancel Contract Button -->
                                                    <button type="button" class="btn btn-link text-danger text-decoration-underline p-0 m-0 btn-sm"
                                                        onclick="confirmCancellation('{{ $status->id }}')">
                                                        Cancel Contract
                                                    </button>
                                                    @endif
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        @endforeach

                        {{-- No Results --}}
                        @if ($EmploymentStatus->isEmpty())
                            <div class="col-12 text-center text-muted">
                                <p>No employment status records found.</p>
                            </div>
                        @endif
                    </div>

                    {{-- Pagination --}}
                    <div class="d-flex justify-content-center mt-3">
                        {{ $EmploymentStatus->appends(request()->query())->links('pagination::bootstrap-4') }}
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- SweetAlert Scripts -->
    <script>
        function confirmAdvance(event, actionText) {
            event.preventDefault();
            const form = event.target.closest('form');
            
            Swal.fire({
                title: 'Confirm Status Update',
                text: `Are you sure you want to ${actionText.toLowerCase()}?`,
                icon: 'question',
                showCancelButton: true,
                confirmButtonColor: '#3085d6',
                cancelButtonColor: '#d33',
                confirmButtonText: 'Yes, proceed',
                cancelButtonText: 'Cancel'
            }).then((result) => {
                if (result.isConfirmed) {
                    form.submit();
                }
            });
        }

        function confirmCancellation(employmentStatusId) {
            Swal.fire({
                title: 'Cancel Contract',
                html: `
                    <form id="cancelForm" action="/employment_status/${employmentStatusId}/cancel" method="POST">
                        @csrf
                        <div class="mb-3">
                            <label for="reason" class="form-label">Reason for Cancellation</label>
                            <textarea name="cancellation_reason" id="reason" class="form-control" rows="4" required placeholder="Enter reason..."></textarea>
                        </div>
                    </form>
                `,
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#d33',
                cancelButtonColor: '#3085d6',
                confirmButtonText: 'Confirm Cancel',
                cancelButtonText: 'Go Back',
                focusConfirm: false,
                preConfirm: () => {
                    const reason = document.getElementById('reason').value;
                    if (!reason) {
                        Swal.showValidationMessage('Please enter a cancellation reason');
                        return false;
                    }
                    return true;
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    document.getElementById('cancelForm').submit();
                }
            });
        }

        function showCancellationDetails(name, status, cancelledBy, cancelledOn, reason) {
            Swal.fire({
                title: 'Cancellation Details',
                html: `
                    <div class="text-left">
                        <div class="mb-3">
                            <h6 class="font-bold text-gray-800">Worker</h6>
                            <p class="text-gray-600">${name}</p>
                        </div>
                        <div class="mb-3">
                            <h6 class="font-bold text-gray-800">Status</h6>
                            <span class="badge bg-danger text-white">${status}</span>
                        </div>
                        <div class="mb-3">
                            <h6 class="font-bold text-gray-800">Cancelled By</h6>
                            <p class="text-gray-600">${cancelledBy}</p>
                        </div>
                        <div class="mb-3">
                            <h6 class="font-bold text-gray-800">Cancelled On</h6>
                            <p class="text-gray-600">${cancelledOn}</p>
                        </div>
                        <div class="mb-3">
                            <h6 class="font-bold text-gray-800">Reason</h6>
                            <p class="text-gray-600">${reason}</p>
                        </div>
                    </div>
                `,
                icon: 'info',
                confirmButtonColor: '#3085d6',
                confirmButtonText: 'Close'
            });
        }

        // Show success message if exists
        @if(session('success'))
            const Toast = Swal.mixin({
                toast: true,
                position: 'center',
                showConfirmButton: false,
                timer: 3000,
                timerProgressBar: true,
                didOpen: (toast) => {
                    toast.addEventListener('mouseenter', Swal.stopTimer)
                    toast.addEventListener('mouseleave', Swal.resumeTimer)
                }
            });

            Toast.fire({
                icon: 'success',
                title: '{{ is_array(session('success')) ? (session('success')['message'] ?? 'Operation completed successfully.') : session('success') }}'
            });
        @endif

        // Show error message if exists
        @if(session('error'))
            Swal.fire({
                title: '{{ is_array(session('error')) ? (session('error')['title'] ?? 'Error!') : 'Error!' }}',
                @if(is_array(session('error')))
                    @if(isset(session('error')['errors']))
                    html: `
                        <div class="text-left">
                            <p>{{ session('error')['message'] ?? 'An error occurred.' }}</p>
                            <ul class="list-disc list-inside mt-2">
                                @foreach(session('error')['errors'] as $field => $errors)
                                    @foreach($errors as $error)
                                        <li>{{ $error }}</li>
                                    @endforeach
                                @endforeach
                            </ul>
                        </div>
                    `,
                    @else
                    text: '{{ session('error')['message'] ?? 'An error occurred.' }}',
                    @endif
                @else
                text: '{{ session('error') }}',
                @endif
                icon: 'error'
            });
        @endif
    </script>
</x-app-layout>