@extends('layouts.workers')

@section('content')
<div class="container mx-auto px-4 py-6">
    <div class="bg-white rounded-xl shadow-md overflow-hidden">
        <!-- Header -->
        <div class="bg-blue-600 px-6 py-4">
            <div class="flex justify-between items-center">
                <div>
                    <h2 class="text-2xl font-bold text-blue">Add New Candidate</h2>
                    <p class="text-blue-100"><strong>Fill in the candidate information below</strong></p>
                </div>
                <a href="{{ route('workers.index') }}" class="text-blue hover:text-blue-200 transition-colors">
                    <i class="fas fa-arrow-left mr-1"></i> Back to List
                </a>
            </div>
        </div>

        <!-- Main Form -->
        <div class="p-6">
            <!-- Success/Error Messages -->
            @if ($errors->any())
                <div class="mb-6 p-4 bg-red-50 border-l-4 border-red-500 text-red-700 rounded">
                    <div class="font-bold">Please fix the following errors:</div>
                    <ul class="mt-2 list-disc list-inside">
                        @foreach ($errors->all() as $error)
                            <li>{{ $error }}</li>
                        @endforeach
                    </ul>
                </div>
            @endif

            @if(session('success'))
                <div class="mb-6 p-4 bg-green-50 border-l-4 border-green-500 text-green-700 rounded">
                    <div class="font-bold">Success!</div>
                    <p>{{ session('success') }}</p>
                </div>
            @endif

            <form action="{{ route('workers.store') }}" method="POST" enctype="multipart/form-data" class="space-y-6" id="workerForm">
                @csrf

                <!-- Profile Photo Upload -->
                <div class="mb-4">
                    <label class="form-label">Profile Photo</label>
                    <div class="d-flex align-items-center">
                        <div class="position-relative me-3">
                            <div class="rounded-lg overflow-hidden" style="width: 150px; height: 150px; background-color: #f8f9fa;">
                                <img id="profilePreview" src="" alt="Profile Preview" class="w-100 h-100 object-fit-cover" style="display: none;">
                                <div id="noImageText" class="w-100 h-100 d-flex align-items-center justify-content-center text-muted">
                                    <i class="fas fa-user fa-3x"></i>
                                </div>
                            </div>
                        </div>
                        <div class="flex-grow-1">
                            <div class="drop-zone" id="photoDropArea" style="border: 2px dashed #dee2e6; border-radius: 4px; padding: 1rem; text-align: center; cursor: pointer;">
                                <input type="file" id="profileInput" name="photo_url" class="d-none" accept=".jpg,.jpeg,.png">
                                <div class="text-muted">
                                    <i class="fas fa-cloud-upload-alt me-2"></i>
                                    <span>Drag & drop or click to upload</span>
                                    </div>
                                <div class="small text-muted mt-1">Allowed formats: JPG, JPEG, PNG (max 2MB)</div>
                            </div>
                        </div>
                    </div>
                    @error('photo_url')
                        <div class="text-danger mt-1">{{ $message }}</div>
                    @enderror
                </div>

                <!-- Personal Information Section -->
                <div class="bg-gray-50 p-6 rounded-lg border border-gray-200">
                    <div class="flex items-center mb-4">
                        <div class="bg-blue-600 w-2 h-8 mr-3 rounded-full"></div>
                        <h3 class="text-xl font-semibold text-gray-800">Personal Information</h3>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Name Fields -->
                        <div>
                            <label for="surname" class="block text-sm font-medium text-gray-700 mb-1">Surname <span class="text-red-500">*</span></label>
                            <input type="text" id="surname" name="surname" value="{{ old('surname') }}"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('surname') border-red-500 @enderror" required>
                            @error('surname')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        <div>
                            <label for="given_names" class="block text-sm font-medium text-gray-700 mb-1">Given Names <span class="text-red-500">*</span></label>
                            <input type="text" id="given_names" name="given_names" value="{{ old('given_names') }}"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('given_names') border-red-500 @enderror" required>
                            @error('given_names')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>

                        <!-- date_of_birth and Gender -->
                        <div>
                            <label for="date_of_birth" class="block text-sm font-medium text-gray-700 mb-1">Date of Birth <span class="text-red-500">*</span></label>
                            <input type="date" id="date_of_birth" name="date_of_birth" value="{{ old('date_of_birth') }}"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('date_of_birth') border-red-500 @enderror"
                                   max="{{ date('Y-m-d') }}" required>
                            @error('date_of_birth')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        <div>
                            <label for="gender" class="block text-sm font-medium text-gray-700 mb-1">Gender <span class="text-red-500">*</span></label>
                            <select id="gender" name="gender" class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('gender') border-red-500 @enderror" required>
                                <option value="" disabled selected>Select Gender</option>
                                <option value="Male" @selected(old('gender') == 'Male')>Male</option>
                                <option value="Female" @selected(old('gender') == 'Female')>Female</option>
                                <option value="Other" @selected(old('gender') == 'Other')>Other</option>
                            </select>
                            @error('gender')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>

                        <!-- Marital Status and Religion -->
                        <div>
                            <label for="marital_status" class="block text-sm font-medium text-gray-700 mb-1">Marital Status <span class="text-red-500">*</span></label>
                            <select id="marital_status" name="marital_status" class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('marital_status') border-red-500 @enderror" required>
                                <option value="" disabled selected>Select Status</option>
                                <option value="Single" @selected(old('marital_status') == 'Single')>Single</option>
                                <option value="Married" @selected(old('marital_status') == 'Married')>Married</option>
                                <option value="Divorced" @selected(old('marital_status') == 'Divorced')>Divorced</option>
                                <option value="Widowed" @selected(old('marital_status') == 'Widowed')>Widowed</option>
                            </select>
                            @error('marital_status')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        <div>
                            <label for="religion" class="block text-sm font-medium text-gray-700 mb-1">Religion <span class="text-red-500">*</span></label>
                            <select id="religion" name="religion" class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('religion') border-red-500 @enderror" required>
                                <option value="" disabled selected>Select Religion</option>
                                <option value="Muslim" @selected(old('religion') == 'Muslim')>Muslim</option>
                                <option value="Non Muslim" @selected(old('religion') == 'Non Muslim')>Non Muslim</option>
                            </select>
                            @error('religion')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>

                        <!-- Occupation and ID -->
                        <div>
                            <label for="occupation" class="block text-sm font-medium text-gray-700 mb-1">Occupation <span class="text-red-500">*</span></label>
                            <select id="occupation" name="occupation" class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500 @error('occupation') border-red-500 @enderror" required>
                                <option value="" disabled selected>Select Occupation</option>
                                <option value="Software Engineer" @selected(old('occupation') == 'Software Engineer')>Software Engineer</option>
                                <option value="Data Analyst" @selected(old('occupation') == 'Data Analyst')>Data Analyst</option>
                                <option value="Product Manager" @selected(old('occupation') == 'Product Manager')>Product Manager</option>
                                <option value="Web Developer" @selected(old('occupation') == 'Web Developer')>Web Developer</option>
                                <option value="HR Manager" @selected(old('occupation') == 'HR Manager')>HR Manager</option>
                                <option value="Marketing Specialist" @selected(old('occupation') == 'Marketing Specialist')>Marketing Specialist</option>
                                <option value="Sales Executive" @selected(old('occupation') == 'Sales Executive')>Sales Executive</option>
                                <option value="Customer Support" @selected(old('occupation') == 'Customer Support')>Customer Support</option>
                            </select>
                            @error('occupation')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        
                        <div>
                            <label for="id_number" class="block text-sm font-medium text-gray-700 mb-1">ID Number <span class="text-red-500">*</span></label>
                            <input type="text" id="id_number" name="id_number" value="{{ old('id_number') }}"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('id_number') border-red-500 @enderror" required>
                            @error('id_number')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Education & Skills Section -->
                <div class="bg-gray-50 p-6 rounded-lg border border-gray-200">
                    <div class="flex items-center mb-4">
                        <div class="bg-blue-600 w-2 h-8 mr-3 rounded-full"></div>
                        <h3 class="text-xl font-semibold text-gray-800">Education & Skills</h3>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Education -->
                        <div>
                            <label for="education_level" class="block text-sm font-medium text-gray-700 mb-1">Education Level <span class="text-red-500">*</span></label>
                            <select id="education_level" name="education_level" class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500 @error('education_level') border-red-500 @enderror" required>
                                <option value="" disabled selected>Select Education Level</option>
                                <option value="High School" @selected(old('education_level') == 'High School')>High School</option>
                                <option value="Associate's Degree" @selected(old('education_level') == 'Associate\'s Degree')>Associate's Degree</option>
                                <option value="Bachelor's Degree" @selected(old('education_level') == 'Bachelor\'s Degree')>Bachelor's Degree</option>
                                <option value="Master's Degree" @selected(old('education_level') == 'Master\'s Degree')>Master's Degree</option>
                                <option value="Doctorate" @selected(old('education_level') == 'Doctorate')>Doctorate</option>
                                <option value="Other" @selected(old('education_level') == 'Other')>Other</option>
                            </select>
                            @error('education_level')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>

                        <!-- Work Experience -->
                        <div>
                            <label for="work_experience" class="block text-sm font-medium text-gray-700 mb-1">Work Experience <span class="text-red-500">*</span></label>
                            <textarea id="work_experience" name="work_experience" rows="3" class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('work_experience') border-red-500 @enderror" required>{{ old('work_experience') }}</textarea>
                            @error('work_experience')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>

                        <!-- Skills -->
                        <div>
                            <label for="skills" class="block text-sm font-medium text-gray-700 mb-1">Skills <span class="text-red-500">*</span></label>
                            <select id="skills" name="skills[]" multiple class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 multiselect @error('skills') border-red-500 @enderror" required>
                                <option value="Cooking" @selected(is_array(old('skills')) && in_array('Cooking', old('skills')))>Cooking</option>
                                <option value="Driving" @selected(is_array(old('skills')) && in_array('Driving', old('skills')))>Driving</option>
                                <option value="Data Analysis" @selected(is_array(old('skills')) && in_array('Data Analysis', old('skills')))>Data Analysis</option>
                                <option value="Cleaning" @selected(is_array(old('skills')) && in_array('Cleaning', old('skills')))>Cleaning</option>
                                <option value="Gardening" @selected(is_array(old('skills')) && in_array('Gardening', old('skills')))>Gardening</option>
                                <option value="Web Development" @selected(is_array(old('skills')) && in_array('Web Development', old('skills')))>Web Development</option>
                            </select>
                            <p class="mt-1 text-xs text-gray-500">Hold Ctrl/Cmd to select multiple</p>
                            @error('skills')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>

                        <!-- Languages -->
                        <div>
                            <label for="languages" class="block text-sm font-medium text-gray-700 mb-1">Languages <span class="text-red-500">*</span></label>
                            <select id="languages" name="languages[]" multiple class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 multiselect @error('languages') border-red-500 @enderror" required>
                                <option value="English" @selected(is_array(old('languages')) && in_array('English', old('languages')))>English</option>
                                <option value="Arabic" @selected(is_array(old('languages')) && in_array('Arabic', old('languages')))>Arabic</option>
                                <option value="French" @selected(is_array(old('languages')) && in_array('French', old('languages')))>French</option>
                                <option value="Swahili" @selected(is_array(old('languages')) && in_array('Swahili', old('languages')))>Swahili</option>
                            </select>
                            <p class="mt-1 text-xs text-gray-500">Hold Ctrl/Cmd to select multiple</p>
                            @error('languages')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Passport Information Section -->
                <div class="bg-gray-50 p-6 rounded-lg border border-gray-200">
                    <div class="flex items-center mb-4">
                        <div class="bg-blue-600 w-2 h-8 mr-3 rounded-full"></div>
                        <h3 class="text-xl font-semibold text-gray-800">Passport Information</h3>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Passport Number -->
                        <div>
                            <label for="passport_number" class="block text-sm font-medium text-gray-700 mb-1">Passport Number <span class="text-red-500">*</span></label>
                            <input type="text" id="passport_number" name="passport_number" value="{{ old('passport_number') }}"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('passport_number') border-red-500 @enderror" required>
                            @error('passport_number')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>

                        <!-- Passport Issue Place -->
                        <div>
                            <label for="passport_issue_place" class="block text-sm font-medium text-gray-700 mb-1">Passport Issue Place <span class="text-red-500">*</span></label>
                            <input type="text" id="passport_issue_place" name="passport_issue_place" value="{{ old('passport_issue_place') }}"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('passport_issue_place') border-red-500 @enderror" required>
                            @error('passport_issue_place')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>

                        <!-- Passport Issue Date -->
                        <div>
                            <label for="passport_issue_date" class="block text-sm font-medium text-gray-700 mb-1">Passport Issue Date <span class="text-red-500">*</span></label>
                            <input type="date" id="passport_issue_date" name="passport_issue_date" value="{{ old('passport_issue_date') }}"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('passport_issue_date') border-red-500 @enderror" required>
                            @error('passport_issue_date')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>

                        <!-- Passport Expiry Date -->
                        <div>
                            <label for="passport_expiry_date" class="block text-sm font-medium text-gray-700 mb-1">Passport Expiry Date <span class="text-red-500">*</span></label>
                            <input type="date" id="passport_expiry_date" name="passport_expiry_date" value="{{ old('passport_expiry_date') }}"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('passport_expiry_date') border-red-500 @enderror" required>
                            @error('passport_expiry_date')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>

                        <!-- Passport File -->
                        <div class="col-span-2">
                            <label for="passport_file" class="block text-sm font-medium text-gray-700 mb-1">Passport File <span class="text-red-500">*</span></label>
                            <input type="file" id="passport_file" name="passport_file" accept=".pdf,.jpeg,.jpg,.png"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('passport_file') border-red-500 @enderror" required>
                            @error('passport_file')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Address Section -->
                <div class="bg-gray-50 p-6 rounded-lg border border-gray-200">
                    <div class="flex items-center mb-4">
                        <div class="bg-blue-600 w-2 h-8 mr-3 rounded-full"></div>
                        <h3 class="text-xl font-semibold text-gray-800">Address Details</h3>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                            <label for="country" class="block text-sm font-medium text-gray-700 mb-1">Country <span class="text-red-500">*</span></label>
                            <input type="text" id="country" name="country" value="{{ old('country') }}"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('country') border-red-500 @enderror" required>
                            @error('country')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        <div>
                            <label for="city" class="block text-sm font-medium text-gray-700 mb-1">City <span class="text-red-500">*</span></label>
                            <input type="text" id="city" name="city" value="{{ old('city') }}"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('city') border-red-500 @enderror" required>
                            @error('city')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        <div>
                            <label for="place_of_birth" class="block text-sm font-medium text-gray-700 mb-1">Place of Birth <span class="text-red-500">*</span></label>
                            <input type="text" id="place_of_birth" name="place_of_birth" value="{{ old('place_of_birth') }}"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('place_of_birth') border-red-500 @enderror" required>
                            @error('place_of_birth')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        <div>
                            <label for="sub_agent_id" class="block text-sm font-medium text-gray-700 mb-1">Sub-Agent</label>
                            <select name="sub_agent_id" id="sub_agent_id" class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('sub_agent_id') border-red-500 @enderror">
                                <option value="">-- Select Sub-Agent --</option>
                                @foreach($subAgents as $subAgent)
                                    <option value="{{ $subAgent->id }}" @selected(old('sub_agent_id') == $subAgent->id)>
                                        {{ $subAgent->name }} ({{ $subAgent->commission }}%)
                                    </option>
                                @endforeach
                            </select>
                            @error('sub_agent_id')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Contact Information Section -->
                <div class="bg-gray-50 p-6 rounded-lg border border-gray-200">
                    <div class="flex items-center mb-4">
                        <div class="bg-blue-600 w-2 h-8 mr-3 rounded-full"></div>
                        <h3 class="text-xl font-semibold text-gray-800">Contact Information</h3>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Mobile Number -->
                        <div>
                            <label for="mobile_number" class="block text-sm font-medium text-gray-700 mb-1">Mobile Number <span class="text-red-500">*</span></label>
                            <input type="tel" id="mobile_number" name="mobile_number" value="{{ old('mobile_number') }}"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('mobile_number') border-red-500 @enderror" required>
                            @error('mobile_number')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>

                        <!-- Email -->
                        <div>
                            <label for="email" class="block text-sm font-medium text-gray-700 mb-1">Email <span class="text-red-500">*</span></label>
                            <input type="email" id="email" name="email" value="{{ old('email') }}"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('email') border-red-500 @enderror" required>
                            @error('email')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Emergency Contact Section -->
                <div class="bg-gray-50 p-6 rounded-lg border border-gray-200">
                    <div class="flex items-center mb-4">
                        <div class="bg-blue-600 w-2 h-8 mr-3 rounded-full"></div>
                        <h3 class="text-xl font-semibold text-gray-800">Emergency Contact</h3>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Relative Name -->
                        <div>
                            <label for="relative_name" class="block text-sm font-medium text-gray-700 mb-1">Relative Name <span class="text-red-500">*</span></label>
                            <input type="text" id="relative_name" name="relative_name" value="{{ old('relative_name') }}"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('relative_name') border-red-500 @enderror" required>
                            @error('relative_name')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>

                        <!-- Relative Kinship -->
                        <div>
                            <label for="relative_kinship" class="block text-sm font-medium text-gray-700 mb-1">Relationship <span class="text-red-500">*</span></label>
                            <select id="relative_kinship" name="relative_kinship" class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('relative_kinship') border-red-500 @enderror" required>
                                <option value="" disabled selected>Select Relationship</option>
                                <option value="Spouse" @selected(old('relative_kinship') == 'Spouse')>Spouse</option>
                                <option value="Parent" @selected(old('relative_kinship') == 'Parent')>Parent</option>
                                <option value="Sibling" @selected(old('relative_kinship') == 'Sibling')>Sibling</option>
                                <option value="Child" @selected(old('relative_kinship') == 'Child')>Child</option>
                                <option value="Other" @selected(old('relative_kinship') == 'Other')>Other</option>
                            </select>
                            @error('relative_kinship')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>

                        <!-- Relative Phone -->
                        <div>
                            <label for="relative_phone" class="block text-sm font-medium text-gray-700 mb-1">Relative Phone <span class="text-red-500">*</span></label>
                            <input type="tel" id="relative_phone" name="relative_phone" value="{{ old('relative_phone') }}"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('relative_phone') border-red-500 @enderror" required>
                            @error('relative_phone')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>

                        <!-- Relative Address -->
                        <div>
                            <label for="relative_address" class="block text-sm font-medium text-gray-700 mb-1">Relative Address <span class="text-red-500">*</span></label>
                            <input type="text" id="relative_address" name="relative_address" value="{{ old('relative_address') }}"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 @error('relative_address') border-red-500 @enderror" required>
                            @error('relative_address')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Acknowledgment Section -->
                <div class="bg-gray-50 p-6 rounded-lg border border-gray-200">
                    <div class="flex items-center mb-4">
                        <div class="bg-blue-600 w-2 h-8 mr-3 rounded-full"></div>
                        <h3 class="text-xl font-semibold text-gray-800">Acknowledgment</h3>
                    </div>

                    <div class="flex items-start">
                        <div class="flex items-center h-5">
                            <input type="checkbox" id="acknowledge" name="acknowledge" value="1" class="w-4 h-4 border border-gray-300 rounded focus:ring-blue-500 focus:ring-2 @error('acknowledge') border-red-500 @enderror" required>
                        </div>
                        <div class="ml-3 text-sm">
                            <label for="acknowledge" class="font-medium text-gray-700">I acknowledge that all the information provided is accurate and true.</label>
                            @error('acknowledge')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Submit Button -->
                <div class="flex justify-end space-x-4">
                    <a href="{{ route('workers.index') }}" class="px-6 py-2 border border-gray-300 rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                        Cancel
                    </a>
                    <button type="submit" id="submitBtn" class="px-6 py-2 border border-transparent rounded-md shadow-sm text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                        <span id="submitText">Create Worker</span>
                        <span id="submitSpinner" class="hidden ml-2">
                            <i class="fas fa-spinner fa-spin"></i>
                        </span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Include Select2 CSS and JS -->
<link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>

<!-- Include SweetAlert2 CSS and JS -->
<link href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css" rel="stylesheet">
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<style>
    .swal2-popup {
        font-size: 0.875rem !important;
        padding: 1rem !important;
    }
    .swal2-title {
        font-size: 1.25rem !important;
    }
    .swal2-html-container {
        font-size: 0.875rem !important;
        margin: 0.5rem 0 !important;
    }
    .swal2-actions {
        margin-top: 0.5rem !important;
    }
    .swal2-confirm {
        padding: 0.5rem 1rem !important;
        font-size: 0.875rem !important;
    }
</style>

<!-- JavaScript for Form Functionality -->
<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Initialize Select2 for multiselect fields
        $('#skills, #languages').select2({
            placeholder: "Select options",
            width: '100%',
            closeOnSelect: false,
            allowClear: true
        });

        // Custom SweetAlert configuration
        const Toast = Swal.mixin({
            toast: true,
            position: 'top-end',
            showConfirmButton: false,
            timer: 3000,
            timerProgressBar: true,
            customClass: {
                popup: 'colored-toast'
            }
        });

        // Profile Photo Upload
        const profileInput = document.getElementById('profileInput');
        const profilePreview = document.getElementById('profilePreview');
        const noImageText = document.getElementById('noImageText');
        const photoDropArea = document.getElementById('photoDropArea');

        function handlePhotoUpload(file) {
            // Validate file type
            const allowedTypes = ['image/jpeg', 'image/png'];
            if (!allowedTypes.includes(file.type)) {
                Toast.fire({
                    icon: 'error',
                    title: 'Invalid file type. Please upload a JPG or PNG image.'
                });
                return;
            }

            // Validate file size (2MB max)
            const maxSize = 2 * 1024 * 1024; // 2MB in bytes
            if (file.size > maxSize) {
                Toast.fire({
                    icon: 'error',
                    title: 'File size exceeds 2MB limit.'
                });
                return;
            }

            // Show preview
            const reader = new FileReader();
            reader.onload = function(e) {
                profilePreview.src = e.target.result;
                profilePreview.style.display = 'block';
                noImageText.style.display = 'none';
            };
            reader.readAsDataURL(file);
        }

        // Handle file selection
        profileInput.addEventListener('change', function(e) {
            if (this.files && this.files[0]) {
                handlePhotoUpload(this.files[0]);
            }
        });

        // Handle drag and drop
        photoDropArea.addEventListener('dragover', function(e) {
            e.preventDefault();
            this.style.borderColor = '#0d6efd';
        });

        photoDropArea.addEventListener('dragleave', function(e) {
            e.preventDefault();
            this.style.borderColor = '#dee2e6';
        });

        photoDropArea.addEventListener('drop', function(e) {
            e.preventDefault();
            this.style.borderColor = '#dee2e6';
            
            if (e.dataTransfer.files && e.dataTransfer.files[0]) {
                handlePhotoUpload(e.dataTransfer.files[0]);
                profileInput.files = e.dataTransfer.files;
            }
        });

        // Handle click to upload
        photoDropArea.addEventListener('click', function() {
            profileInput.click();
        });

        // Passport Document Upload
        const passportInput = document.getElementById('passportInput');
        const passportPreview = document.getElementById('passportPreview');
        const noPassportText = document.getElementById('noPassportText');
        const passportPdfPreview = document.getElementById('passportPdfPreview');
        const pdfFileName = document.getElementById('pdfFileName');
        const passportDropArea = document.getElementById('passportDropArea');

        function handlePassportUpload(file) {
            // Validate file type
            const allowedTypes = ['image/jpeg', 'image/png', 'application/pdf'];
            if (!allowedTypes.includes(file.type)) {
                Toast.fire({
                    icon: 'error',
                    title: 'Invalid file type. Please upload a JPG, PNG, or PDF file.'
                });
                return;
            }

            // Validate file size (5MB max)
            const maxSize = 5 * 1024 * 1024; // 5MB in bytes
            if (file.size > maxSize) {
                Toast.fire({
                    icon: 'error',
                    title: 'File size exceeds 5MB limit.'
                });
                return;
            }

            // Hide "no file" text and show appropriate preview
            noPassportText.style.display = 'none';

            if (file.type === 'application/pdf') {
                // Handle PDF file
                passportPreview.style.display = 'none';
                passportPdfPreview.style.display = 'flex';
                pdfFileName.textContent = file.name.length > 20 
                    ? file.name.substring(0, 17) + '...' 
                    : file.name;
            } else {
                // Handle image file
                passportPdfPreview.style.display = 'none';
                const reader = new FileReader();
                reader.onload = function(e) {
                    passportPreview.src = e.target.result;
                    passportPreview.style.display = 'block';
                };
                reader.readAsDataURL(file);
            }
        }

        // Handle file selection
        passportInput.addEventListener('change', function(e) {
            if (this.files && this.files[0]) {
                handlePassportUpload(this.files[0]);
            }
        });

        // Handle drag and drop
        passportDropArea.addEventListener('dragover', function(e) {
            e.preventDefault();
            this.style.borderColor = '#0d6efd';
        });

        passportDropArea.addEventListener('dragleave', function(e) {
            e.preventDefault();
            this.style.borderColor = '#dee2e6';
        });

        passportDropArea.addEventListener('drop', function(e) {
            e.preventDefault();
            this.style.borderColor = '#dee2e6';
            
            if (e.dataTransfer.files && e.dataTransfer.files[0]) {
                handlePassportUpload(e.dataTransfer.files[0]);
                passportInput.files = e.dataTransfer.files;
            }
        });

        // Handle click to upload
        passportDropArea.addEventListener('click', function() {
            passportInput.click();
        });

        // Passport Date Validations
        const passportIssueDate = document.getElementById('passport_issue_date');
        const passportExpiryDate = document.getElementById('passport_expiry_date');
        const passportNumber = document.getElementById('passport_number');

        function validatePassportDates() {
            const issueDate = new Date(passportIssueDate.value);
            const expiryDate = new Date(passportExpiryDate.value);
            const today = new Date();
            const minIssueDate = new Date();
            minIssueDate.setFullYear(today.getFullYear() - 10);

            // Reset error states
            passportIssueDate.classList.remove('border-red-500');
            passportExpiryDate.classList.remove('border-red-500');

            let isValid = true;
            let errorMessage = '';

            // Validate issue date
            if (!passportIssueDate.value) {
                passportIssueDate.classList.add('border-red-500');
                errorMessage = 'Passport issue date is required';
                isValid = false;
            } else if (issueDate > today) {
                passportIssueDate.classList.add('border-red-500');
                errorMessage = 'Passport issue date cannot be in the future';
                isValid = false;
            } else if (issueDate < minIssueDate) {
                passportIssueDate.classList.add('border-red-500');
                errorMessage = 'Passport issue date cannot be more than 10 years ago';
                isValid = false;
            }

            // Validate expiry date
            if (!passportExpiryDate.value) {
                passportExpiryDate.classList.add('border-red-500');
                errorMessage = 'Passport expiry date is required';
                isValid = false;
            } else if (expiryDate <= issueDate) {
                passportExpiryDate.classList.add('border-red-500');
                errorMessage = 'Passport expiry date must be after issue date';
                isValid = false;
            } else if (expiryDate <= today) {
                passportExpiryDate.classList.add('border-red-500');
                errorMessage = 'Passport expiry date must be in the future';
                isValid = false;
            }

            // Validate passport number format
            if (passportNumber && passportNumber.value) {
                const passportRegex = /^[A-Z0-9]{6,9}$/;
                if (!passportRegex.test(passportNumber.value)) {
                    passportNumber.classList.add('border-red-500');
                    errorMessage = 'Passport number must be 6-9 characters long and contain only uppercase letters and numbers';
                    isValid = false;
                } else {
                    passportNumber.classList.remove('border-red-500');
                }
            }

            if (!isValid) {
                Swal.fire({
                    icon: 'error',
                    title: 'Passport Information Error',
                    text: errorMessage,
                    customClass: {
                        popup: 'swal2-popup',
                        title: 'swal2-title',
                        htmlContainer: 'swal2-html-container',
                        confirmButton: 'swal2-confirm'
                    }
                });
            }

            return isValid;
        }

        // Add event listeners for date changes
        passportIssueDate.addEventListener('change', validatePassportDates);
        passportExpiryDate.addEventListener('change', validatePassportDates);
        if (passportNumber) {
            passportNumber.addEventListener('input', function() {
                this.value = this.value.toUpperCase();
                validatePassportDates();
            });
        }

        // Set max date for issue date to today
        passportIssueDate.max = new Date().toISOString().split('T')[0];
        
        // Set min date for expiry date to tomorrow
        const tomorrow = new Date();
        tomorrow.setDate(tomorrow.getDate() + 1);
        passportExpiryDate.min = tomorrow.toISOString().split('T')[0];

        // Update expiry date min when issue date changes
        passportIssueDate.addEventListener('change', function() {
            const minExpiryDate = new Date(this.value);
            minExpiryDate.setDate(minExpiryDate.getDate() + 1);
            passportExpiryDate.min = minExpiryDate.toISOString().split('T')[0];
            
            if (passportExpiryDate.value && new Date(passportExpiryDate.value) <= minExpiryDate) {
                passportExpiryDate.value = minExpiryDate.toISOString().split('T')[0];
            }
        });

        // Form validation and submission
        const workerForm = document.getElementById('workerForm');
        if (workerForm) {
            workerForm.addEventListener('submit', async function(e) {
                e.preventDefault();
                console.log('Form submission started'); // Debug log

                // Show loading state
                const submitBtn = document.getElementById('submitBtn');
                const submitText = document.getElementById('submitText');
                const submitSpinner = document.getElementById('submitSpinner');

                if (submitBtn && submitText && submitSpinner) {
                    submitBtn.disabled = true;
                    submitText.textContent = 'Processing...';
                    submitSpinner.classList.remove('hidden');
                }

                try {
                    // Validate passport dates first
                    if (!validatePassportDates()) {
                        resetSubmitButton();
                        return;
                    }

                    // Validate required fields
                    const requiredFields = workerForm.querySelectorAll('[required]');
                    let isValid = true;
                    let firstInvalidField = null;
                    
                    requiredFields.forEach(field => {
                        if (field.type === 'checkbox') {
                            if (!field.checked) {
                                field.closest('.flex').classList.add('border-red-500', 'p-2', 'rounded');
                                isValid = false;
                                if (!firstInvalidField) firstInvalidField = field;
                            } else {
                                field.closest('.flex').classList.remove('border-red-500', 'p-2', 'rounded');
                            }
                        } else if (field.type === 'select-multiple') {
                            const select = $(field);
                            if (!select.val() || select.val().length === 0) {
                                select.next('.select2-container').addClass('border-red-500');
                                isValid = false;
                                if (!firstInvalidField) firstInvalidField = field;
                            } else {
                                select.next('.select2-container').removeClass('border-red-500');
                            }
                        } else if (!field.value.trim()) {
                            field.classList.add('border-red-500');
                            isValid = false;
                            if (!firstInvalidField) firstInvalidField = field;
                        } else {
                            field.classList.remove('border-red-500');
                        }
                    });

                    if (!isValid) {
                        resetSubmitButton();
                        Swal.fire({
                            icon: 'error',
                            title: 'Required Fields Missing',
                            text: 'Please fill in all required fields.',
                            customClass: {
                                popup: 'swal2-popup',
                                title: 'swal2-title',
                                htmlContainer: 'swal2-html-container',
                                confirmButton: 'swal2-confirm'
                            },
                            didOpen: () => {
                                if (firstInvalidField) {
                                    firstInvalidField.scrollIntoView({ behavior: 'smooth', block: 'center' });
                                    firstInvalidField.focus();
                                }
                            }
                        });
                        return;
                    }

                    // Submit form
                    const formData = new FormData(workerForm);
                    console.log('Submitting form...'); // Debug log

                    const response = await fetch(workerForm.action, {
                        method: 'POST',
                        body: formData,
                        headers: {
                            'X-Requested-With': 'XMLHttpRequest',
                            'Accept': 'application/json'
                        }
                    });

                    console.log('Response received:', response.status); // Debug log

                    const contentType = response.headers.get('content-type');
                    if (contentType && contentType.includes('application/json')) {
                        const data = await response.json();
                        console.log('Response data:', data); // Debug log

                        if (response.ok) {
                            Swal.fire({
                                icon: 'success',
                                title: 'Success',
                                text: data.message || 'Worker added successfully!'
                            }).then(() => {
                                window.location.href = data.redirect || '/workers';
                            });
                        } else {
                            if (data.errors) {
                                const errorList = Object.values(data.errors).flat().map(error => `<li>${error}</li>`).join('');
                                Swal.fire({
                                    icon: 'error',
                                    title: 'Validation Errors',
                                    html: `<ul class="text-left list-disc list-inside text-sm">${errorList}</ul>`,
                                    customClass: {
                                        popup: 'swal2-popup',
                                        title: 'swal2-title',
                                        htmlContainer: 'swal2-html-container',
                                        confirmButton: 'swal2-confirm'
                                    }
                                });
                            } else {
                                Swal.fire({
                                    icon: 'error',
                                    title: 'Error',
                                    text: data.message || 'An error occurred while saving the worker.'
                                });
                            }
                        }
                    } else {
                        // Handle non-JSON response (redirect)
                        window.location.href = response.url;
                    }
                } catch (error) {
                    console.error('Error:', error); // Debug log
                    Swal.fire({
                        icon: 'error',
                        title: 'Error',
                        text: error.message || 'An error occurred while saving the worker.'
                    });
                } finally {
                    resetSubmitButton();
                }
            });
        }

        function resetSubmitButton() {
            const submitBtn = document.getElementById('submitBtn');
            const submitText = document.getElementById('submitText');
            const submitSpinner = document.getElementById('submitSpinner');

            if (submitBtn && submitText && submitSpinner) {
                submitBtn.disabled = false;
                submitText.textContent = 'Create Worker';
                submitSpinner.classList.add('hidden');
            }
        }
    });
</script>
@endsection